package termopl;

import java.util.LinkedList;

public abstract class Template 
{
	
	public static final int NO_QUANTIFIER = 0;
	public static final int ZERO_OR_ONE   = 1;
	public static final int ZERO_OR_MORE  = 2;
	public static final int ONE_OR_MORE   = 3;
	
	public static Template adjp = ADJP();
	
	public int quantifier;
	public boolean computeBaseForm;
	public boolean head;
	public Tester tester;
	
	public Template()
	{
		quantifier = NO_QUANTIFIER;
		computeBaseForm = false;
		head = false;
		tester = null;
	}
	
	public Template(boolean computeBaseForm)
	{
		quantifier = NO_QUANTIFIER;
		this.computeBaseForm = computeBaseForm;
		head = false;
		tester = null;
	}
	
	public Template(int quantifier)
	{
		this.quantifier = quantifier;
		computeBaseForm = false;
		head = false;
		tester = null;
	}
	
	public Template(int quantifier, boolean computeBaseForm)
	{
		this.quantifier = quantifier;
		this.computeBaseForm = computeBaseForm;
		head = false;
		tester = null;
	}
	
	public void setQuantifier(int quantifier)
	{
		this.quantifier = quantifier;
	}
	
	public void setTester(Tester tester)
	{
		this.tester = tester;
	}
	
	public void setComputeBaseForm(boolean computeBaseForm)
	{
		this.computeBaseForm = computeBaseForm;
	}
	
	public void setHead(boolean head)
	{
		this.head = head;
	}
	
	public boolean test(Token token, Tagset tagset)
	{
		if (tester != null) return tester.test(token, tagset);
		return true;
	}
	
	public boolean test(MatchedFragment matched, Tagset tagset)
	{
		if (tester != null) return tester.test(matched, tagset);
		return true;
	}
	
	public abstract LinkedList<Template> getElements();
	
	public static boolean testPOS(Token token, Tagset tagset, String... POS)
	{
		String pos = token.getPOS(tagset);
		
		for (String s : POS) {
			if (pos.equals(s)) return true;
		}
		return false;
	}
	
	public static boolean testForm(Token token, String form)
	{
		return token.form.equals(form);
	}
	
	public static boolean testLemma(Token token, String lemma)
	{
		return token.lemma.equals(lemma);
	}
	
	public static boolean testAgreement(MatchedFragment matched, Tagset tagset)
	{
		return testAgreement(matched, tagset, null, null, null);
	}
	
	public static boolean testAgreement(MatchedFragment matched, Tagset tagset, String gcase, String gender, String number)
	{
		LinkedList<MatchedToken> tokens = matched.getMatchedTokens();
		int size = tokens.size();
		String t_gcase, t_gender, t_number;
		
		if (size == 1) {
			Token t = tokens.getFirst().token;
			
			t_gcase = tagset.getCategory(t, "case");
			t_gender = tagset.getCategory(t, "gender");
			t_number = tagset.getCategory(t, "number");
			if (t_gcase != null && t_gender != null && t_number != null)
				return agree(gcase, t_gcase, gender, t_gender, number, t_number);
		}
		else {
			for (MatchedToken mt : tokens) {
				Token t = mt.token;
				
				t_gcase = tagset.getCategory(t, "case");
				t_gender = tagset.getCategory(t, "gender");
				t_number = tagset.getCategory(t, "number");
				if (t_gcase != null && t_gender != null && t_number != null) {
					if (!agree(gcase, t_gcase, gender, t_gender, number, t_number)) return false;
					if (gcase == null) gcase = t_gcase;
					if (gender == null) gender = t_gender;
					if (number == null) number = t_number;
				}
			}
		}
		return true;
	}
	
	public static boolean agree(String c1, String c2, String g1, String g2, String n1, String n2)
	{
		if (c1 != null && !c2.equals(c1)) return false;
		if (g1 != null && !g2.equals(g1)) return false;
		if (n1 != null && !n2.equals(n1)) return false;
		return true;
	}
	
	public static Template adj()
	{
		SimpleTemplate st = new SimpleTemplate() {
			public boolean test(Token token, Tagset tagset)
			{
				return testPOS(token, tagset, "adj");
			}
		};
		return st;
	}
	
	public static Template ppas()
	{
		SimpleTemplate st = new SimpleTemplate() {
			public boolean test(Token token, Tagset tagset)
			{
				return testPOS(token, tagset, "ppas");
			}
		};
		return st;
	}
	
	public static Template noun()
	{
		SimpleTemplate st = new SimpleTemplate(true) {
			public boolean test(Token token, Tagset tagset)
			{
				return testPOS(token, tagset, "subst", "ger");
			}
		};
		return st;
	}
	
	public static Template adja()
	{
		SimpleTemplate st = new SimpleTemplate() {
			public boolean test(Token token, Tagset tagset)
			{
				return testPOS(token, tagset, "adja");
			}
		};
		return st;
	}
	
	public static Template dash()
	{
		SimpleTemplate st = new SimpleTemplate() {
			public boolean test(Token token, Tagset tagset)
			{
				return testLemma(token, "-");
			}
		};
		return st;
	}
	
	public static Template adja_adj()
	{
		return new AndTemplate(adja(), dash(), adj());
	}
	
	public static Template ADJ_star()
	{
		return new OrTemplate(ZERO_OR_MORE, adj(), ppas(), adja_adj());
	}
	
	public static Template NAP()
	{
		AndTemplate at = new AndTemplate(true, ADJ_star(), noun(), ADJ_star()) {
			public boolean test(MatchedFragment matched, Tagset tagset)
			{
				return testAgreement(matched, tagset);
			}
		};
		return at;
	}
	
	public static Template NAP_gen_star()
	{
		AndTemplate at = new AndTemplate(ZERO_OR_MORE, ADJ_star(), noun(), ADJ_star()) {
			public boolean test(MatchedFragment matched, Tagset tagset)
			{
				return testAgreement(matched, tagset, "gen", null, null);
			}
		};
		return at;
	}
	
	public static Template NPP()
	{
		return new AndTemplate(NAP(), NAP_gen_star());
	}
	
	public static Template ADJP()
	{
		return new OrTemplate(adj(), ppas(), adja_adj());
	}
	
}
