package termopl;

import java.util.*;

@SuppressWarnings("serial")
public class TermEx extends Term
{
	public String id;
	private LinkedList<String> parents;
	private LinkedList<String> children;
	private LinkedList<String> equivTerms;
	private LinkedList<Pair<String, LinkedList<WordReplacement>>> relatedTerms;
	
	public TermEx(String str, int len)
	{
		super(str, len);
		this.id = str;
		parents = children = equivTerms = null;
		relatedTerms = null;
	}
	
	public void addChild(TermEx child, HashMap<String, Term> termMap)
	{
		if (child != this) {
			if (equivalent(child)) addEquivalentTerm(child, termMap);
			else {
				if (children == null) children = new LinkedList<String>();
				if (child.parents == null) child.parents = new LinkedList<String>();
				if (!children.contains(child.id)) {
					children.add(child.id);
					child.parents.add(this.id);
				}
			}
		}
	}
	
	public boolean removeChild(TermEx child)
	{
		if (children != null) {
			if (children.remove(child.id)) {
				child.parents.remove(id);
				return true;
			}
		}
		return false;
	}
	
	public boolean equivalent(TermEx term)
	{
		if (len != term.len) return false;
		
		LinkedList<String> ph1 = new LinkedList<String>(Arrays.asList(str.split(" ")));
		String[] ph2 = term.str.split(" ");
		
		for (String s2 : ph2) {
			if (!ph1.remove(s2)) return false;
		}
		return ph1.isEmpty();
	}
	
	public void addEquivalentTerm(TermEx term, HashMap<String, Term> termMap)
	{
		if (term != this) {
			if (equivTerms == null) {
				equivTerms = new LinkedList<String>();
				equivTerms.add(id);
			}
			if (term.equivTerms != null) {
				for (String e : term.equivTerms) {
					if (!equivTerms.contains(e)) {
						TermEx t = (TermEx)termMap.get(e);
						
						equivTerms.add(e);
						t.equivTerms = equivTerms;
					}
				}
			}
			else {
				if (!equivTerms.contains(term.id)) {
					equivTerms.add(term.id);
					term.equivTerms = equivTerms;
				}
			}
			for (String e : equivTerms) {
				if (e != id) {
					if (parents != null) {
						for (String s : parents) {
							TermEx p = (TermEx)termMap.get(s);
							TermEx t = (TermEx)termMap.get(e);
							
							p.removeChild(t);
						}
					}
				}
			}
			removeDescendant(this, termMap);
		}
	}
	
	public void relateWith(TermEx term, LinkedList<WordReplacement> replacements)
	{
		addRelatedTerm(term, replacements);
		
		LinkedList<WordReplacement> newReplacements = new LinkedList<WordReplacement>();
		
		for (WordReplacement wr : replacements) {
			WordReplacement newWR = new WordReplacement(wr.word, wr.expression);
			
			if (wr.relTypes != null)
				for (Integer relType : wr.relTypes) newWR.add(-relType);
			newReplacements.add(newWR);
		}
		term.addRelatedTerm(this, newReplacements);
	}
	
	public void addRelatedTerm(TermEx term, LinkedList<WordReplacement> replacements)
	{
		if (relatedTerms == null) {
			relatedTerms = new LinkedList<Pair<String, LinkedList<WordReplacement>>>();
		}
		if (!containedInRelated(term, relatedTerms)) 
			relatedTerms.add(new Pair<String, LinkedList<WordReplacement>>(term.id, replacements));
	}
	
	public boolean isDescendant(TermEx term, HashMap<String, Term> termMap)
	{
		if (term == this) return true;
		if (equivTerms != null) {
			for (String t : equivTerms)
				if (term.id == t) return true;
		}
		if (children != null) {
			for (String c : children) {
				TermEx t = (TermEx)termMap.get(c);
				
				if (t.isDescendant(term, termMap)) return true;
			}
			return false;
		}
		return false;
	}
	
	
	public void removeDescendant(TermEx term, HashMap<String, Term> termMap)
	{
		if (children != null) {
			removeChild(term);
			if (equivTerms != null) {
				for (String e : equivTerms) {
					TermEx t = (TermEx)termMap.get(e);
					
					if (t.children != null) t.children.remove(term.id);
				}
			}
		}
	}
	
	public LinkedList<String> collectParents(HashMap<String, Term> termMap)
	{
		if (equivTerms == null) return parents;
		else {
			LinkedList<String> allParents = new LinkedList<String>();
			
			for (String e : equivTerms) {
				TermEx t = (TermEx)termMap.get(e);
				LinkedList<String> plist = t.getParents();
				
				if (plist != null) {
					for (String p : plist) {
						if (!allParents.contains(p)) allParents.add(p);
					}
				}
			}
			return allParents;
		}
	}
	
	public LinkedList<String> collectChildren(HashMap<String, Term> termMap)
	{
		LinkedList<String> allChildren = new LinkedList<String>();
		
		if (equivTerms == null) allChildren = children;
		else {
			allChildren = new LinkedList<String>();
			for (String e : equivTerms) {
				TermEx t = (TermEx)termMap.get(e);
				LinkedList<String> clist = t.getChildren();
				
				if (clist != null) {
					for (String c : clist) {
						TermEx tc = (TermEx)termMap.get(c);
						
						if (tc.equivTerms != null) {
							boolean found = false;
							
							for (String ec : tc.equivTerms) {
								if (allChildren.contains(ec)) {
									found = true;
									break;
								}
							}
							if (!found) allChildren.add(c);
						}
						else if (!allChildren.contains(c)) allChildren.add(c);
					}
				}
			}
		}
		return allChildren;
	}
	
	public LinkedList<Pair<String, LinkedList<WordReplacement>>> collectRelated(HashMap<String, Term> termMap)
	{
		if (equivTerms == null) return relatedTerms;
		else {
			LinkedList<Pair<String, LinkedList<WordReplacement>>> allRelated = new LinkedList<Pair<String, LinkedList<WordReplacement>>>();
			
			for (String e : equivTerms) {
				TermEx t = (TermEx)termMap.get(e);
				LinkedList<Pair<String, LinkedList<WordReplacement>>> rlist = t.getRelatedTerms();
				
				if (rlist != null) {
					for (Pair<String, LinkedList<WordReplacement>> r : rlist) {
						if (!contains(allRelated, r)) allRelated.add(r);
					}
				}
			}
			return allRelated;
		}
	}
	
	public boolean contains(LinkedList<Pair<String, LinkedList<WordReplacement>>> list, Pair<String, LinkedList<WordReplacement>> e)
	{
		for (Pair<String, LinkedList<WordReplacement>> p : list) {
			if (p.first == e.first) return true;
		}
		return false;
	}
	
	public LinkedList<String> getParents()
	{
		return parents;
	}
	
	public LinkedList<String> getChildren()
	{
		return children;
	}
	
	public LinkedList<String> getEquivalentTerms()
	{
		return equivTerms;
	}
	
	public LinkedList<Pair<String, LinkedList<WordReplacement>>> getRelatedTerms()
	{
		return relatedTerms;
	}
	
	public void clean()
	{
		parents = children = equivTerms = null;
		relatedTerms = null;
	}
	
	public static boolean containedInRelated(TermEx term, LinkedList<Pair<String, LinkedList<WordReplacement>>> relatedTerms) 
	{
		if (relatedTerms == null) return false;
		for (Pair<String, LinkedList<WordReplacement>> p : relatedTerms) {
			if (p.first == term.id) return true;
		}
		return false;
	}
	
}
