 package termopl;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.io.*;
import java.text.Collator;
import java.util.*;
import java.util.List;
import java.util.concurrent.*;

import javax.swing.*;
import pl.sgjp.morfeusz.*;

public class TermoPLDocument extends Commander
{

	public static final String EXTRACT_OPTIONS[] 	= { "All Selected", "Recently selected", "None" };
	public static final int MAX_TERM_QUEUE_SIZE 	= 512;
	public static final int MAX_THREADS         	= Runtime.getRuntime().availableProcessors();
	public static final int TERMS_WINDOW 	 		= 1;
	public static final int FORMS_WINDOW 	 		= 2;
	public static final int SENTENCES_WINDOW 		= 3;
	public static final int GROUPS_WINDOW			= 4;
	public static final int EXPORT_RESULTS			= 1;
	public static final int EXPORT_FORMS			= 2;
	public static final int EXPORT_SENTENCES		= 3;
	public static final int EXPORT_GROUPS   		= 4;
	public static final int SAVE_THREAD				= 0;
	public static final int EXPORT_RESULTS_THREAD	= 1;
	public static final int EXPORT_FORMS_THREAD		= 2;
	public static final int EXPORT_SENTENCES_THREAD	= 3;
	public static final int EXPORT_GROUPS_THREAD	= 4;
	public static final int MAX_SAVE_THREADS		= 5;
	
	private static final Term DUMMY_TERM = new Term();
	
	public static int newCounter = 0;
	
	private LinkedList<TermoPLWindow> windows;
	private TermsView termsView;
	private FormsView formsView;
	private SentencesView sentencesView;
	private GroupsView groupsView;
	private ColumnSorter sorter;
	private BlockingQueue<Term> termQueue;
	private CountDownLatch countDown;
	private ExtractorEngine extractor;
	private BaseFormGuesser baseFormGuesser;
	private File input;
	private BatchParameters bp;
	private Preferences preferences;
	private File[] selectedFiles;
	private File[] oldFiles;
	private File[] newFiles;
	private Term[] terms;
	private LinkedList<FileDescr> analyzedFiles;
	private HashMap<String, Term> cterms;
	private String corpusName;
	private String docName;
	private String docDisplayName;
	private int tableSize;
	private int nsentences;
	private int ntokens;
	private int nterms;
	private int rebuild;
	private int acceptDET;
	private boolean loaded;
	private boolean modified;
	private boolean cancelled;
	private boolean corrupted;
	private boolean formsCollected;
	private boolean sentencesIndexed;
	private boolean reuseTaggedFiles;
	private boolean searching;

	transient private HashMap<String, Term> termMap;
	transient private HashMap<String, LinkedList<TermEx>> termIndex;
	transient private HashMap<String, LinkedList<WordReplacement>> wordIndex;
	transient private WordNet wordNet;
	transient private Morfeusz morfeusz;
	transient private Thread[] saveThreads;
	transient private boolean interrupted;

	@SuppressWarnings("serial")
	public AbstractAction selectAction = new AbstractAction("Select") {
		public void actionPerformed(ActionEvent e) 
		{
			selectFiles();
		}
	};
	@SuppressWarnings("serial")
	public AbstractAction extractAction = new AbstractAction("Extract") {
		public void actionPerformed(ActionEvent e) 
		{
			extract();
		}
	};
	@SuppressWarnings("serial")
	public AbstractAction compareAction = new AbstractAction("Compare") {
		public void actionPerformed(ActionEvent e) 
		{
			compare();
		}
	};

	public TermoPLDocument(TermoPL superCommander)
	{
		super(superCommander);
		input = null;
		bp = null;
		preferences = (Preferences)TermoPL.preferences.clone();
		sorter = new ColumnSorter();
		windows = new LinkedList<TermoPLWindow>();
		termsView = null;
		formsView = null;
		sentencesView = null;
		groupsView = null;
		selectedFiles = null;
		oldFiles = null;
		newFiles = null;
		analyzedFiles = null;
		terms = null;
		termMap = null;
		cterms = TermoPL.cterms;
		termIndex = null;
		wordNet = null;
		morfeusz = null;
		saveThreads = new Thread[MAX_SAVE_THREADS];
		extractor = null;
		baseFormGuesser = null;
		corpusName = null;
		docName = "New " + ++newCounter;
		docDisplayName = docName;
		tableSize = 0;
		nsentences = 0;
		ntokens = 0;
		nterms = 0;
		rebuild = 0;
		acceptDET = -1;
		loaded = true;
		modified = false;
		cancelled = false;
		interrupted = false;
		corrupted = false;
		formsCollected = false;
		sentencesIndexed = false;
		reuseTaggedFiles = preferences.reuseTaggedFiles;
		searching = false;
		createWindow(TERMS_WINDOW);
	}
	
	public TermoPLDocument(TermoPL superCommander, File file) 
	{
		super(superCommander);
		input = file;
		bp = null;
		preferences = TermoPL.preferences;
		sorter = new ColumnSorter();
		windows = new LinkedList<TermoPLWindow>();
		termsView = null;
		formsView = null;
		sentencesView = null;
		groupsView = null;
		selectedFiles = null;
		oldFiles = null;
		newFiles = null;
		analyzedFiles = null;
		terms = null;
		termMap = null;
		cterms = null;
		termIndex = null;
		wordNet = null;
		morfeusz = null;
		saveThreads = new Thread[MAX_SAVE_THREADS];
		extractor = null;
		baseFormGuesser = null;
		docName = input.getName();
		docDisplayName = TermoPL.getDisplayName(docName);
		tableSize = 0;
		rebuild = 0;
		acceptDET = -1;
		loaded = false;
		modified = false;
		cancelled = false;
		interrupted = false;
		corrupted = false;
		formsCollected = false;
		sentencesIndexed = false;
		reuseTaggedFiles = preferences.reuseTaggedFiles;
		searching = false;
		if (!TermoPL.batchMode) {
			createWindow(TERMS_WINDOW);
			termsView.setAccessory(null);
			termsView.showProgress(true);
			termsView.pleaseWait();
		}
		
		LoadThread loadThread = new LoadThread(input);
		loadThread.start();
		
		if (TermoPL.batchMode) {
			try {
				loadThread.join();
			}
			catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
	}
	
	public TermoPLDocument(TermoPL superCommander, BatchParameters bp) 
	{
		super(superCommander);
		input = null;
		this.bp = bp;
		preferences = TermoPL.preferences;
		sorter = new ColumnSorter();
		windows = null;
		termsView = null;
		formsView = null;
		sentencesView = null;
		groupsView = null;
		selectedFiles = null;
		oldFiles = null;
		newFiles = null;
		analyzedFiles = null;
		terms = null;
		termMap = null;
		cterms = null;
		termIndex = null;
		wordNet = null;
		morfeusz = null;
		saveThreads = new Thread[MAX_SAVE_THREADS];
		extractor = null;
		baseFormGuesser = null;
		corpusName = TermoPL.getFileName(bp.inputFiles[0]);
		docName = null;
		docDisplayName = null;
		tableSize = 0;
		rebuild = 0;
		acceptDET = -1;
		loaded = true;
		modified = false;
		cancelled = false;
		interrupted = false;
		corrupted = false;
		formsCollected = false;
		sentencesIndexed = false;
		reuseTaggedFiles = preferences.reuseTaggedFiles;//??????
		searching = false;
	}
	
	public void finishLoading()
	{
		if (!TermoPL.batchMode) {
			if (isLoaded()) {
				boolean missing = false;
				
				for (File f : selectedFiles) {
					if (!f.exists()) {
						missing = true;
						break;
					}
				}
				if (missing) selectFiles();
				if (termsView != null) {
					if (preferences.extract) {
						termsView.setWarning(TermsView.EXTRACT_WARNING);
						termsView.setAccessory(extractAction);
					}
					else if (preferences.compare) {
						termsView.setWarning(TermsView.COMPARE_WARNING);
						termsView.setAccessory(compareAction);
					}
				}
			}
			if (terms != null) {
				if (terms[0] instanceof TermEx) {
					termMap = new HashMap<String, Term>();
					for (Term t : terms) {
						TermEx tex = (TermEx)t;
						
						termMap.put(tex.id, t);
					}
				}
			}
		}
	}
	
	public String getCorpusName()
	{
		return corpusName;
	}
	
	public String getDocumentName()
	{
		return docName;
	}
	
	public String getDocumentDisplayName()
	{
		return docDisplayName;
	}
	
	public String getPath()
	{
		if (input != null) return input.getPath();
		return null;
	}
	
	public void waitForSaveThreadsToFinish()
	{
		for (Thread th : saveThreads) {
			if (th != null) {
				try {
					th.join();
				} 
				catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	public void startBatchProcessing()
	{
		File file;

		if (bp.wrk == null) preferences.workSpace = "";
		else {
			if (bp.wrk.endsWith(File.separator)) {
				int i = bp.wrk.lastIndexOf(File.separator);
				
				if (i > 0) 
					bp.wrk = bp.wrk.substring(0, bp.wrk.length() - 
						File.separator.length());
			}
			preferences.workSpace = bp.wrk;
		}
		if (bp.wdn != null) {
			if (bp.wdn.endsWith(File.separator)) {
				int i = bp.wdn.lastIndexOf(File.separator);
				
				if (i > 0) 
					bp.wdn = bp.wdn.substring(0, bp.wdn.length() - 
						File.separator.length());
			}
			preferences.wordNetPath = bp.wdn;
			preferences.makeGroups = true;
			preferences.useWordNet = true;
		}
		if (bp.inputFiles == null) {
			System.out.println("No input");
			return;
		}
		else setSelectedFiles(bp.inputFiles);
		if (bp.comp != null) {
			file = new File(bp.comp);
			if (file.exists()) {
				cterms = loadContrastiveTerms(file);
				if (cterms == null) {
					System.out.println("Can't read comparative data");
					return;
				}
				else preferences.applyContrastiveRanking = true;
			}
			else {
				System.out.println(fileDoesNotExist(bp.comp));
				return;
			}
		}
		preferences.reuseTaggedFiles = false;
		if (bp.sw != null) {
			if (bp.sw != null) {
				LinkedList<String> sw = null;
				
				try {
					BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(bp.sw), "UTF8"));
					String line;
					
					do {
						line = reader.readLine();
						if (line != null) {
							line = line.trim();
							if (!line.isEmpty()) {
								if (sw == null) sw = new LinkedList<String>();
								if (!sw.contains(line)) sw.add(line);
							}
						}
					} while (line != null);
					reader.close();
					preferences.stopWords = sw;
					preferences.checkStopWords = true;
				}
				catch (IOException exception) {
					exception.printStackTrace();
					preferences.stopWords = null;
					preferences.checkStopWords = false;
				}
			}
			else {
				preferences.stopWords = null;
				preferences.checkStopWords = false;
			}
		}
		else {
			preferences.stopWords = null;
			preferences.checkStopWords = false;
		}
		if (bp.cp != null) {
			if (bp.cp != null) {
				LinkedList<String> cp = null;
				
				try {
					BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(bp.cp), "UTF8"));
					String line;
					
					do {
						line = reader.readLine();
						if (line != null) {
							line = line.trim();
							if (!line.isEmpty()) {
								if (cp == null) cp = new LinkedList<String>();
								if (!cp.contains(line)) cp.add(line);
							}
						}
					} while (line != null);
					reader.close();
					if (cp != null) {
						PParser parser = new PParser(cp, null);
						
						parser.parse();
						preferences.removeCompoundPreps = true;
						preferences.compPreps = cp;
						preferences.compoundPrepositions = parser.getTemplate();
					}
					else {
						preferences.removeCompoundPreps = false;
						preferences.compPreps = null;
						preferences.compoundPrepositions = null;
					}
				}
				catch (IOException exception) {
					preferences.removeCompoundPreps = false;
					preferences.compPreps = null;
					preferences.compoundPrepositions = null;
				}
			}
			else {
				preferences.removeCompoundPreps = false;
				preferences.compPreps = null;
				preferences.compoundPrepositions = null;
			}
		}
		else {
			preferences.removeCompoundPreps = false;
			preferences.compPreps = null;
			preferences.compoundPrepositions = null;
		}
		if (bp.ct != null) {
			if (bp.ct != null) {
				LinkedList<CommonTerm> ct = null;
				
				try {
					BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(bp.ct), "UTF8"));
					String line;
					
					do {
						line = reader.readLine();
						if (line != null) {
							line = line.trim();
							if (!line.isEmpty()) {
								String bf = null, sf = null;
								
								if (ct == null) ct = new LinkedList<CommonTerm>();
								
								int index = line.indexOf("[");
								
								if (index < 0) {
									sf = line;
									bf = line;
								}
								else {
									sf = line.substring(0, index).trim();
									bf = line.substring(index + 1).replace("]", "").trim();
								}
								ct.add(new CommonTerm(bf, sf));
							}
						}
					} while (line != null);
					reader.close();
					preferences.commonTerms = ct;
					preferences.removeCommonTerms = true;
				}
				catch (IOException exception) {
					exception.printStackTrace();
					preferences.commonTerms = null;
					preferences.removeCommonTerms = false;
				}
			}
			else {
				preferences.commonTerms = null;
				preferences.removeCommonTerms = false;
			}
		}
		else {
			preferences.commonTerms = null;
			preferences.removeCommonTerms = false;
		}
		if (bp.language != null) {
			preferences.language = bp.language;
		}
		else preferences.language = "pl";
		if (bp.method != null) {
			if (bp.method.equals("ud")) preferences.useUD = true;
			else preferences.useUD = false;
		}
		else preferences.useUD = false;
		if (bp.ignoreCase != null) {
			if (bp.ignoreCase.equals("yes")) preferences.ignoreCase = true;
			else preferences.ignoreCase = false;
		}
		else preferences.ignoreCase = true;
		if (bp.det != null) {
			if (bp.det.equals("include")) preferences.detHandling = 1;
			else if (bp.det.equals("article")) preferences.detHandling = 2;
			else if (bp.det.equals("exclude")) preferences.detHandling = 0;
			else preferences.detHandling = 1; 
		}
		else preferences.detHandling = 1;
		if (bp.detRatio == -1) {
			preferences.detectDeterminers = false;
		}
		else {
			preferences.detectDeterminers = true;
			preferences.detRatio = bp.detRatio;
		}
		if (bp.tagset != null) {
			if (bp.tagset != null) {
				TParser parser = new TParser(new File(bp.tagset), null);
				Tagset tmp;
				
				parser.parse();
				tmp = parser.getTagset();
				if (tmp != null) {
					preferences.useCustomTagset = true;
					preferences.tagset = tmp;
				}
				else return;
			}
			else {
				System.out.println(fileDoesNotExist(bp.tagset));
				return;
			}
		}
		else {
			preferences.useCustomTagset = false;
			preferences.tagset = Tagset.createDefaultTagset() ;
		}
		if (bp.grammar != null) {
			if (bp.grammar != null) {
				GParser parser = new GParser(new File(bp.grammar), preferences.getTagset(), null);
				Template tmp;
				
				parser.parse();
				tmp = parser.getTemplate();
				if (tmp != null) {
					preferences.useCustomGrammar = true;
					preferences.template = tmp;
				}
				else return;
			}
			else {
				System.out.println(fileDoesNotExist(bp.grammar));
				return;
			}
		}
		else {
			preferences.useCustomGrammar = false;
			preferences.template = Template.NPP();
		}
		if (bp.mw > 0) preferences.multiWordTermsOnly = true;
		if (bp.sf > 0) preferences.calculateBaseForms = false;
		if (bp.nf > 0) preferences.collectAllForms = false;
		if (bp.exportForms != null) preferences.collectAllForms = true;
		if (bp.indx > 0) preferences.makeIndex = true;
		if (bp.group > 0) preferences.makeGroups = true;
		if (bp.exportSentences != null) preferences.makeIndex = true;
		if (preferences.makeIndex) preferences.reuseTaggedFiles = true;
		if (bp.srch > 0) {
			if (bp.srch < 4) {
				preferences.useNPMIMethod = true;
				preferences.NPMIMethod = bp.srch;
			}
			else {
				preferences.useNPMIMethod = false;
				if (bp.srch == 4) preferences.trimFromLeftToRight = false;
				else preferences.trimFromLeftToRight = true;
			}
		}
		if (bp.cntx > 0) preferences.cntxMethod = bp.cntx;
		if (bp.sort > 0) {
			int asc = (bp.sort > 8 ? 1 : -1);
			
			if (bp.sort > 8) bp.sort -= 8;
			preferences.sortedColumn = bp.sort;
			preferences.sortPrefs[bp.sort] = asc;
		}
		if (bp.tr > 0) {
			preferences.trimResults = true;
			preferences.maxResults = bp.tr;
		}
		if (bp.pf >= 0) {
			preferences.NPMIfactor = bp.pf;
			preferences.useNPMIMethod = true;
			preferences.NPMIMethod = 3;
		}
		if (bp.cc >= 0) {
			if (bp.cc == 0) preferences.applyContrastiveRanking = false;
			else {
				preferences.applyContrastiveRanking = true;
				preferences.contrastiveRankingMethod = bp.cc;
				if (bp.cc == 4) preferences.useCValues = false;
			}
		}
		if (bp.frq > 0) preferences.useCValues = false;
		if (bp.freq >= 0) {
			preferences.applyContrastiveRankingForFrequentTerms = true;
			preferences.minfrq = bp.freq;
		}
		if (bp.cval >= 0) {
			preferences.applyContrastiveRankingForTopRankedTerms = true;
			preferences.mincvalue = bp.cval;
		}
		if (bp.saveOptions != null && bp.saveOptions.length > 0) {
			preferences.saveCount = false;
			preferences.saveRank = false;
			preferences.saveSF = false;
			preferences.saveBF = false;
			preferences.saveCV = false;
			preferences.saveComp = false;
			preferences.saveLen = false;
			preferences.saveFreqs = false;
			preferences.saveFreqin = false;
			preferences.saveContext = false;
			for (String str : bp.saveOptions) {
				if (str.equals("#")) preferences.saveCount = true;
				else if (str.equals("rank")) preferences.saveRank = true;
				else if (str.equals("sf")) preferences.saveSF = true;
				else if (str.equals("bf")) preferences.saveBF = true;
				else if (str.equals("cvalue")) preferences.saveCV = true;
				else if (str.equals("comp")) preferences.saveComp = true;
				else if (str.equals("length")) preferences.saveLen = true;
				else if (str.equals("freq_s")) preferences.saveFreqs = true;
				else if (str.equals("freq_in")) preferences.saveFreqin = true;
				else if (str.equals("context")) preferences.saveContext = true;
			}
			if (!preferences.saveSF && !preferences.saveBF) {
				if (preferences.calculateBaseForms) preferences.saveBF = true;
				else preferences.saveSF = true;
			}
		}
		
		extractor = new ExtractorEngine(this, selectedFiles);
		
		GroupingEngine groupingEngine = null;
		
		if (preferences.makeGroups) groupingEngine = new GroupingEngine();
		
		Finalizer finalizer = new Finalizer();
		ExtractorThread et = new ExtractorThread(extractor, groupingEngine, finalizer, null);
		
		terms = null;
		termMap = null;
		et.start();
		try {
			et.join();
		} 
		catch (InterruptedException e) {
			e.printStackTrace();
		}
	}
	
	public void setSelectedFiles(String[] inputFiles)
	{
		LinkedList<File> sf = new LinkedList<File>();
		
		for (String p : inputFiles) {
			addSelectedFile(p, sf);
		}
		if (sf.isEmpty()) System.out.println("No input");
		else selectedFiles = sf.toArray(new File[0]);
	}
	
	public void addSelectedFile(String p, LinkedList<File> sf)
	{
		int index = p.lastIndexOf(File.separator);
		String dirName, fileName;
		File dir;
		File[] files = null;
		
		if (index < 0) {
			fileName = p;
			dirName = null;
		}
		else {
			if (index < p.length() - 1) {
				fileName = p.substring(index + 1);
				dirName = p.substring(0, index);
			}
			else {
				fileName = null;
				dirName = p;
			}
		}
		if (fileName == null) {
			dir = new File(dirName);
			if (dir.exists()) sf.add(dir);
			else System.out.println(fileDoesNotExist(p));
		}
		else {
			if (dirName != null) {
				dir = new File(dirName);
				if (dir.exists()) files = dir.listFiles(new FileNameFilter(fileName));
				else System.out.println(fileDoesNotExist(p));
			}
			else {
				dir = new File("").getAbsoluteFile();
				files = dir.listFiles(new FileNameFilter(fileName));
			}
			if (files != null)
				for (File f : files) sf.add(f);
		}
	}
	
	public String fileDoesNotExist(String f)
	{
		return "File: '" + f + "' does not exist";
	}
	
	public void executeCommand(int commandID, int modifiers)
	{
		switch (commandID) {
			case Command.MERGE_COMMAND:
				merge();
				break;
			case Command.CLOSE_COMMAND:
				close();
				break;
			case Command.SAVE_COMMAND:
				save();
				break;
			case Command.SAVE_AS_COMMAND:
				saveAs();
				break;
			case Command.WORKSPACE_COMMAND:
				selectWorkspace();
				break;
			case Command.EXPORT_COMMAND:
				export(EXPORT_RESULTS);
				break;
			case Command.EXPORT_FORMS_COMMAND:
				export(EXPORT_FORMS);
				break;
			case Command.EXPORT_SENTENCES_COMMAND:
				export(EXPORT_SENTENCES);
				break;
			case Command.EXPORT_GROUPS_COMMAND:
				export(EXPORT_GROUPS);
				break;
			case Command.EXTRACT_COMMAND:
				extract();
				break;
			case Command.COMPARE_COMMAND:
				compare();
				break;
			case Command.SELECT_FILES_COMMAND:
				selectFiles();
				break;
			case Command.SELECT_CONTRASTIVE_TERMS_COMMAD:
				selectContrastiveTerms();
				break;
			case Command.BASE_FORMS_COMMAND:
				preferences.calculateBaseForms = !preferences.calculateBaseForms;
				super.executeCommand(commandID, modifiers);
				if (terms != null) {
					Thread th = new Thread() {
						public void run()
						{
							recalculateForms();
						}
					};
					th.start();
				}
				modified = true;
				break;
			case Command.COLLECT_COMMAND:
				super.executeCommand(commandID, modifiers);
				allowCollectingForms(!preferences.collectAllForms);
				break;
			case Command.INDEX_COMMAND:
				super.executeCommand(commandID, modifiers);
				allowIndexingSentences(!preferences.makeIndex);
				break;
			case Command.PREFERENCES_COMMAND:
				changePreferences();
				break;
			case Command.FORMS_COMMAND:
				if (formsView == null) createWindow(FORMS_WINDOW);
				else getWindow(FORMS_WINDOW).select();
				break;
			case Command.SENTENCES_COMMAND:
				if (sentencesView == null) createWindow(SENTENCES_WINDOW);
				else getWindow(SENTENCES_WINDOW).select();
				break;
			case Command.GROUPS_COMMAND:
				if (groupsView == null) createWindow(GROUPS_WINDOW);
				else getWindow(GROUPS_WINDOW).select();
				break;
			default: super.executeCommand(commandID, modifiers);
		}
	}
	
	public void findMenuStatus(JMenu menu)
	{
	}
	
	public void findCommandStatus(JMenuItem item)
	{
		Command command = (Command)item.getAction();
		boolean b;
		
		switch (command.getCommandID()) {
			case Command.CLOSE_COMMAND:
			case Command.SELECT_FILES_COMMAND:
				item.setEnabled(!searching);
				break;
			case Command.SAVE_COMMAND:
				b = saveThreads[SAVE_THREAD] == null || !saveThreads[SAVE_THREAD].isAlive();
				item.setEnabled(b && modified && terms != null && !searching);
				break;
			case Command.SAVE_AS_COMMAND:
				b = saveThreads[SAVE_THREAD] == null || !saveThreads[SAVE_THREAD].isAlive();
				item.setEnabled(b && terms != null && !searching);
				break;
			case Command.EXPORT_COMMAND:
				b = saveThreads[EXPORT_RESULTS_THREAD] == null || !saveThreads[EXPORT_RESULTS_THREAD].isAlive();
				item.setEnabled(b && terms != null && !searching);
				break;
			case Command.MERGE_COMMAND:
			case Command.COMPARE_COMMAND:
				item.setEnabled(terms != null && !searching);
				break;
			case Command.EXTRACT_COMMAND:
				item.setEnabled(selectedFiles != null || newFiles != null && !searching);
				break;
			case Command.BASE_FORMS_COMMAND:
				item.setSelected(preferences.calculateBaseForms);
				item.setEnabled(!preferences.useCustomTagset && !searching);
				break;
			case Command.COLLECT_COMMAND:
				item.setSelected(preferences.collectAllForms);
				item.setEnabled(!searching);
				break;
			case Command.INDEX_COMMAND:
				item.setSelected(preferences.makeIndex);
				item.setEnabled(!searching);
				break;
			case Command.EXPORT_FORMS_COMMAND:
				b = saveThreads[EXPORT_FORMS_THREAD] == null || !saveThreads[EXPORT_FORMS_THREAD].isAlive();
				item.setEnabled(b && terms != null && formsCollected && !searching);
				break;
			case Command.FORMS_COMMAND:
				item.setEnabled(terms != null && formsCollected && !searching);
				break;
			case Command.EXPORT_SENTENCES_COMMAND:
				b = saveThreads[EXPORT_SENTENCES_THREAD] == null || !saveThreads[EXPORT_SENTENCES_THREAD].isAlive();
				item.setEnabled(b && terms != null && sentencesIndexed && !searching);
				break;
			case Command.SENTENCES_COMMAND:
				item.setEnabled(terms != null && sentencesIndexed && !searching);
				break;
			case Command.EXPORT_GROUPS_COMMAND:
				b = saveThreads[EXPORT_GROUPS_THREAD] == null || !saveThreads[EXPORT_GROUPS_THREAD].isAlive();
				item.setEnabled(b && terms != null && preferences.makeGroups && !searching);
				break;
			case Command.GROUPS_COMMAND:
				item.setEnabled(terms != null && preferences.makeGroups && !searching);
				break;
			default: super.findCommandStatus(item);
		}
	}
	
	public void changePreferences()
	{
		Options options = new Options(preferences);
		int res = options.doDialog();
		boolean setinfo = true;
		
		options.dispose();
		if (res == Options.OK) {
			if (preferences.isModified()) {
				modified = true;
				if (terms != null) {
					if (preferences.extract) {
						rebuild = 2;
						if (termsView != null) {
							termsView.setWarning(TermsView.EXTRACT_WARNING);
							termsView.setAccessory(extractAction);
							setinfo = false;
						}
					}
					else if (preferences.compare) {
						if (termsView != null) {
							termsView.setWarning(TermsView.COMPARE_WARNING);
							termsView.setAccessory(compareAction);
							setinfo = false;
						}
					}
					else if (preferences.repaint) {
						if (cterms != null)	{
							applyContrastiveRanking();
							if (termsView != null) termsView.setInfo(getStat());
						}
					}
					if (preferences.recalculate) {
						calculate(preferences.cntxMethod);
					}
					if (!preferences.applyContrastiveRanking) {
						if (termsView != null) {
							if (setinfo) termsView.setInfo(getStat());
							termsView.refreshColumns();
						}
					}
				}
				if (termsView != null) termsView.repaint();
				preferences.save();
			}
		}
	}
	
	public void extract()
	{
		File[] searchFiles = null;
		int s = 0;
		
		if (rebuild == 0) rebuild = 2;
		else if (rebuild == 1) {
			if (selectedFiles == null) searchFiles = newFiles;
			else {
				int res = JOptionPane.showOptionDialog(TermoPL.dialogOwner,
													  "You may choose to run extraction process from scratch,\n" +
													  "or augment the current list of terms using the recently\n" +
													  "selected files only.\n\n" +
													  "Which set of files are you going to use for term extraction?",
													  "Select source for term extraction",
													  JOptionPane.YES_NO_CANCEL_OPTION,
													  JOptionPane.QUESTION_MESSAGE,
													  null,
													  EXTRACT_OPTIONS,
													  EXTRACT_OPTIONS[1]);
				
				if (res == JOptionPane.CANCEL_OPTION) return;
				if (res == JOptionPane.NO_OPTION) searchFiles = newFiles;
				else rebuild = 2;
			}
			acceptDET = -1;
		}
		if (rebuild == 2) {
			if (oldFiles != null) s += oldFiles.length;
			if (newFiles != null) s += newFiles.length;
			searchFiles = new File[s];
			s = 0;
			if (oldFiles != null)
				for (int i = 0; i < oldFiles.length; i++) searchFiles[s++] = oldFiles[i];
			if (newFiles != null)
				for (int i = 0; i < newFiles.length; i++) searchFiles[s++] = newFiles[i];
			terms = null;
			nsentences = 0;
			ntokens = 0;
			nterms = 0;
			acceptDET = -1;
		}
		if (preferences.makeGroups) {
			if (terms != null) {
				for (Term t : terms) {
					t.str = calcSimplifiedForm(t);
				}
				if (terms[0] instanceof TermEx) {
					for (Term t : terms) {
						((TermEx)t).clean();
					}
				}
				else {
					for (int i = 0; i < terms.length; i++) {
						terms[i] = new TermEx(terms[i].str, terms[i].len);
					}
				}
			}
		}
		else {
			if (terms != null) {
				if (terms[0] instanceof TermEx) {
					for (int i = 0; i < terms.length; i++) {
						terms[i] = new TermEx(terms[i].str, terms[i].len);
					}
				}
			}
		}
		
		ContrastiveDataLoader loader = null;
		Finalizer finalizer = new Finalizer();
		
		modified = true;
		cancelled = false;
		interrupted = false;
		extractor = new ExtractorEngine(this, searchFiles);
		formsCollected = false;
		sentencesIndexed = false;
		if (preferences.applyContrastiveRanking) {
			if (cterms == null) {
				if (preferences.contrastiveDataPath == null) 
					preferences.contrastiveDataPath = TermoPL.selectContrastiveCorpus();
				
				if (preferences.contrastiveDataPath != null)
					loader = new ContrastiveDataLoader(preferences.contrastiveDataPath);
			}
		}
		if (termsView != null) {
			if (rebuild == 2) {
				termsView.resetResults();
				termsView.clearSearch();
			}
			termsView.setAccessory(null);
			termsView.showProgress(true);
		}
		if (formsView != null) formsView.reset();
		if (sentencesView != null) sentencesView.reset();
		if (groupsView != null) groupsView.reset();
		System.gc();
		
		GroupingEngine groupingEngine = null;
		
		if (preferences.makeGroups) groupingEngine = new GroupingEngine();
		
		ExtractorThread et = new ExtractorThread(extractor, groupingEngine, finalizer, loader);
		
		et.start();
	}
	
	public void makeGroups()
	{
		int col = preferences.sortedColumn;
		int sortPref = preferences.sortPrefs[TermoPLConsts.LENGTH];

		// Sort all terms. Shorter terms first.
		preferences.sortedColumn = TermoPLConsts.LENGTH;
		preferences.sortPrefs[TermoPLConsts.LENGTH] = 1;
		sortTable();
		preferences.sortedColumn = col;
		preferences.sortPrefs[TermoPLConsts.LENGTH] = sortPref;

		// Map all words to terms in which they appear.
		mapWordsToTerms();
		
		// Build hierarchy of terms
		makeHierarchy();
		if (preferences.useWordNet && preferences.language.equals("pl") && !isCancelled()) {
			initBaseFormGuesser();
			analizeTermsWithWordnet();
		}
		
		// Cleanup
		termIndex = null;
		baseFormGuesser = null;
	}
	
	public void makeHierarchy()
	{
		int i = 0;
		float progress = 0.0F;
		
		changeProgress(6);
		report(progress);
		for (Term term : terms) {
			if (isCancelled()) break;
			TermEx termEx = (TermEx)term;
			
			// Collect all terms containing head of the current one.
			LinkedList<TermEx> simTerms = termsWithSimilarHead(termEx);
			
			// Build hierarchy
			if (simTerms != null) createTermHierarchy(termEx, simTerms);
			if (++i % 250 == 0) {
				progress = (float)i/((float)terms.length);
				report(progress);
			}
		}
		
		report(1.0F);
	}
	
	public void mapWordsToTerms()
	{
		termIndex = new HashMap<String, LinkedList<TermEx>>();
		
		for (Term term : terms) {
			String[] words = term.str.split(" ");
			
			for (String w : words) {
				LinkedList<TermEx> list = termIndex.get(w);
				
				if (list == null) {
					list = new LinkedList<TermEx>();
					list.add((TermEx)term);
					termIndex.put(w, list);
				}
				else list.add((TermEx)term);
			}
		}
	}
	
	public LinkedList<TermEx> termsWithSimilarHead(TermEx term)
	{
		String[] head = getHeadPhrase(term).second;
		LinkedList<TermEx> simTerms = null;
		
		// Collect all terms containing head of the current one.
		for (String str : head) {
			LinkedList<TermEx> hrel = new LinkedList<TermEx>();
			LinkedList<TermEx> sub = null;
			LinkedList<TermEx> related = termIndex.get(str);
			
			if (related != null) {
				hrel.addAll(related);
				hrel.remove(term);
				for (TermEx t : hrel) {
					if (term.len < t.len) {
						String[] hp = getHeadPhrase(t).second;
						
						if (hp != null) {
							for (String s : hp) {
								if (s.equals(str)) {
									if (sub == null) sub = new LinkedList<TermEx>();
									sub.add(t);
									break;
								}
							}
						}
					}
				}
				if (sub == null) {
					simTerms = null;
					break;
				}
				else {
					if (simTerms == null) simTerms = sub;
					else {
						ListIterator<TermEx> it = simTerms.listIterator();
						
						while (it.hasNext()) {
							TermEx t = it.next();
							
							if (!sub.contains(t)) it.remove();
						}
					}
				}
			}
		}
		return simTerms;
	}
	
	public void createTermHierarchy(TermEx term, LinkedList<TermEx> simTerms)
	{
		for (TermEx t : simTerms) addTermToHierarchy(t, term);
	}
	
	public boolean addTermToHierarchy(TermEx term, TermEx parent)
	{
		if (isLessSpecific(parent, term)) {
			LinkedList<String> children = parent.getChildren();
			boolean found = false;
			
			if (children != null) {
				children = new LinkedList<String>(children);
				for (String t : children) {
					if (addTermToHierarchy(term, (TermEx)termMap.get(t))) found = true;
				}
			}
			if (!found) parent.addChild(term, termMap);
			return true;
		}
		return false;
	}
	
	// Check if t1 is less specific than t2 (or identical to t2)
	public boolean isLessSpecific(TermEx t1, TermEx t2)
	{
		String[] ph1 = t1.str.split(" ");
		String[] ph2 = t2.str.split(" ");
		
		if (contains(ph2, ph1)) {
			Pair<String,String[]> p1 = getHeadPhrase(t1);
			Pair<String,String[]> p2 = getHeadPhrase(t2);
			String[] h1 = p1.second;
			String[] h2 = p2.second;
			
			if (contains(h2, h1)) {
				String s1 = p1.first;
				String s2 = p2.first;
				
				if (s1 != null && s1.equals(s2)) {
					TermEx cntx1 = getContextTerm(t1);
					TermEx cntx2 = getContextTerm(t2);
					
					if (cntx1 == null && cntx2 == null) return true;
					if (cntx1 == null) return true;
					if (cntx2 == null) return false;
					return isLessSpecific(cntx1, cntx2);
				}
			}
		}
		return false;
	}
	
	// Check if t1 contains t2
	public boolean contains(String[] t1, String[] t2)
	{
		if (t1 == null) return false;
		if (t2 == null) return true;
		if (t1.length < t2.length) return false;
		
		LinkedList<String> list = new LinkedList<String>(Arrays.asList(t1));
		
		for (String s2 : t2) {
			if (!list.remove(s2)) return false;
		}
		return true;
	}
	
	// Check if t1 and t2 contain the same words
	public boolean same(String[] t1, String[] t2)
	{
		if (t1 == null && t2 == null) return true;
		if (t1 == null || t2 == null) return false;
		if (t1.length != t2.length) return false;
		
		LinkedList<String> list = new LinkedList<String>(Arrays.asList(t1));
		
		for (String s2 : t2) list.remove(s2);
		return list.isEmpty();
	}
	
	// Get words of term t. If type = 0, get all words. If type = 1, get words from head phrase.
	// If type = 2. get all words that do not belong to head phrase.
	public Pair<String,String[]> getPhrase(Term t, int type)
	{
		Form frm = t.getForms().getFirst();
		LinkedList<String> ph = new LinkedList<String>();
		String w = null;
		String h = null;
		int prevIndex = -1;
		boolean headfound = false;
		
		for (MatchedToken mt : frm.getTokens()) {
			if (type == 0 ||
				(type == 1 && mt.computeBaseForm) ||
				(type == 2 && !mt.computeBaseForm))
			{
				Token tok = mt.token;
				
				if (mt.head() && type == 1) headfound = true;
				if (w == null) w = tok.lemma;
				else w += tok.lemma;
				if (tok instanceof UDToken) {
					int index = ((UDToken)tok).index;
					
					if (tok.spaceAfter || index > prevIndex + 1) {
						if (h == null && headfound) h = w;
						ph.add(w);
						w = null;
						headfound = false;
					}
					prevIndex = index;
				}
				else if (tok instanceof MultiWordToken) {
					int index = ((MultiWordToken)tok).getTokens().getLast().index;
					
					if (tok.spaceAfter || index > prevIndex + 1) {
						if (h == null && headfound) h = w;
						ph.add(w);
						w = null;
						headfound = false;
					}
					prevIndex = index;
				}
				else if (tok.spaceAfter) {
					if (h == null && headfound) h = w;
					ph.add(w);
					w = null;
					headfound = false;
				}
			}
		}
		if (w != null) ph.add(w);
		if (ph.size() > 0) return new Pair<String,String[]>(h, ph.toArray(new String[0]));
		return null;
	}
	
	public String[] getTags(Term t)
	{
		Form frm = t.getForms().getFirst();
		LinkedList<String> tags = new LinkedList<String>();
		
		for (MatchedToken mt : frm.getTokens()) {
			Token tok = mt.token;
			
			if (preferences.useCustomTagset || (!tok.ctag.equals("adja") || tok.form.equals("-"))) {
				tags.add(tok.ctag);
			}
		}
		if (tags.size() > 0) return tags.toArray(new String[0]);
		return null;
	}
	
	public String[] getPhrase(Term t)
	{
		Pair<String,String[]> p = getPhrase(t, 0);
		
		if (p != null) return p.second;
		return null;
	}
	
	public Pair<String,String[]> getHeadPhrase(Term t)
	{
		return getPhrase(t, 1);
	}
	
	public String[] getContextPhrase(Term t)
	{
		Pair<String,String[]> p = getPhrase(t, 2);
		
		if (p != null) return p.second;
		return null;
	}
	
	public TermEx getContextTerm(Term t)
	{
		String[] cntx = getContextPhrase(t);
		
		if (cntx == null) return null;
		return getTerm(cntx, 0, cntx.length);
	}
	
	public TermEx getTerm(String[] words, int start, int end)
	{
		if (start == end) return null;
		
		String str = "";
		Term t;
		
		for (int i = start; i < end; i++) str += words[i] + " ";
		str = str.trim();
		t = termMap.get(str);
		if (t == null) t = getTerm(words, start + 1, end);
		if (t == null) t = getTerm(words, start, end - 1);
		return (TermEx)t;
	}
	
	public void initBaseFormGuesser()
	{
		if (preferences.language.equals("pl") && !preferences.useCustomTagset)
			baseFormGuesser = new BaseFormGuesser(preferences.tagset);
		else baseFormGuesser = new BaseFormGuesser();
	}
	
	public void analizeTermsWithWordnet()
	{
		int i = 0;
		float progress = 0.0F;
		
		wordNet = WordNet.getWordNet(preferences.wordNetPath);
		if (wordNet == null) {
			changeProgress(7);
			wordNet = WordNet.createWordNet(this, preferences.wordNetPath);
		}
		if (wordNet != null) {
			morfeusz = Morph.getAnalyzer();
			wordIndex = new HashMap<String, LinkedList<WordReplacement>>();
			changeProgress(8);
			report(progress);
			for (Term t : terms) {
				if (isCancelled()) break;
				analyzeWithWordnet((TermEx) t);
				if (++i % 1000 == 0) {
					progress = (float)i/(float)terms.length;
					report(progress);
					System.gc();
				}
			}
			report(1.0F);
			wordNet = null;
			morfeusz = null;
			wordIndex = null;
			System.gc();
		}
	}
	
	public void analyzeWithWordnet(TermEx term)
	{
		// Find all words from term
		String[] ph = getPhrase(term);
		
		if (ph != null) {
			String[] tags = getTags(term);
			String[] wdn = new String[ph.length];
			@SuppressWarnings("unchecked")
			LinkedList<WordReplacement>[] wordReplacements = new LinkedList[ph.length];
			boolean makeReplacements = false;
		
			// Make them acceptable for WordNet
			for (int i = 0; i < ph.length; i++) {
				wdn[i] = ph[i];
				if (tags[i].startsWith("subst")) tags[i] = WordNet.NOUN;
				else if (tags[i].startsWith("adj")) tags[i] = WordNet.ADJECTIVE;
				else if (tags[i].startsWith("adv")) tags[i] = WordNet.ADVERB;
				else if (tags[i].startsWith("fin")) tags[i] = WordNet.VERB;
				else if (tags[i].startsWith("praet")) tags[i] = WordNet.VERB;
				else if (tags[i].startsWith("impt")) tags[i] = WordNet.VERB;
				else if (tags[i].startsWith("imps")) tags[i] = WordNet.VERB;
				else if (tags[i].startsWith("inf")) tags[i] = WordNet.VERB;
				else if (tags[i].startsWith("ger")) {
					wdn[i] = baseFormGuesser.nominalForm(ph[i], tags[i]);
					tags[i] = WordNet.NOUN;
				}
				else if (tags[i].startsWith("ppas")) {
					wdn[i] = baseFormGuesser.nominalForm(ph[i], tags[i]);
					tags[i] = WordNet.ADJECTIVE;
				}
				else if (tags[i].startsWith("pact")) {
					wdn[i] = baseFormGuesser.nominalForm(ph[i], tags[i]);
					tags[i] = WordNet.ADJECTIVE;
				}
				else tags[i] = null;
			}
			
			for (int i = 0; i < ph.length; i++) {
				if (tags[i] != null) {
					String key = ph[i] + "_" + tags[i];
					LinkedList<WordReplacement> substList = wordIndex.get(key);
					
					if (substList == null) {
						substList = getWordReplacements(ph[i], wdn[i], tags[i]);
						wordReplacements[i] = substList;
						if (substList == null) substList = new LinkedList<WordReplacement>();
						else substList.add(new WordReplacement(ph[i], ph[i], WordNet.EXACT));
						wordIndex.put(key,  substList);
					}
					else {
						if (substList.isEmpty()) wordReplacements[i] = null;
						else wordReplacements[i] = substList;
					}
				}
				else wordReplacements[i] = null;
			}
			
			for (LinkedList<WordReplacement> list : wordReplacements) {
				if (list != null) {
					makeReplacements = true;
					break;
				}
			}
			if (makeReplacements) {
				for (int i = 0; i < ph.length; i++) {
					if (wordReplacements[i] == null) {
						wordReplacements[i] = new LinkedList<WordReplacement>();
						wordReplacements[i].add(new WordReplacement(ph[i], ph[i], WordNet.EXACT));
					}
				}
				
				LinkedList<TermReplacement> relatedTerms = findRelatedTerms(term, ph, wordReplacements);
				
				if (relatedTerms != null) associate(term, relatedTerms);
			}
		}
	}
	
	public LinkedList<WordReplacement> getWordReplacements(String word, String wordNetForm, String tag) 
	{
		LinkedList<WordReplacement> substList = null;
		LinkedList<WordReplacement> related = wordNet.getRelatedWords(wordNetForm, tag);
		
		if (related != null) {
			for (WordReplacement item : related) {
				String[] expr = item.expression.split(" ");
				LinkedList<LinkedList<String>> nodes = null;
				
				for (String w : expr) {
					LinkedList<String> lemmas = getLemmas(w);
					
					if (lemmas == null) {
						nodes = null;
						break;
					}
					if (nodes == null) nodes = new LinkedList<LinkedList<String>>();
					nodes.add(lemmas);
				}
				if (nodes != null) {
					LinkedList<WordReplacement> list = new LinkedList<WordReplacement>();
					LinkedList<String> lemmas = cartesianProduct(nodes);
					
					for (String lemma : lemmas) {
						if (!lemma.equals(word)) 
							WordReplacement.addReplacement(word, lemma, item.relTypes, list);
					}
					if (!list.isEmpty()) {
						if (substList == null) substList = list;
						else substList.addAll(list);
					}
				}
			}
		}
		return substList;
	}
	
	public LinkedList<String> getLemmas(String word)
	{
		List<MorphInterpretation> interps = morfeusz.analyseAsList(word);
		LinkedList<String> lemmas = getLemmas(0, interps);
		ListIterator<String> it = lemmas.listIterator();
		
		while (it.hasNext()) {
			String lemma = it.next();
			
			if (!termIndex.containsKey(lemma)) it.remove();
		}
		return lemmas;
	}
	
	public LinkedList<String> getLemmas(int start, List<MorphInterpretation> interps)
	{
		int end = -1;
		LinkedList<String> lemmas = new LinkedList<String>();
		LinkedList<String> list = new LinkedList<String>();
		Tagset tagset = preferences.tagset;
		
		for (MorphInterpretation interp : interps) {
			int s = interp.getStartNode();
			int e = interp.getEndNode();
			
			if (s == start) {
				if (e > end) {
					if (end > 0) {
						lemmas.addAll(combine(list, getLemmas(end, interps)));
						list = new LinkedList<String>();
					}
					end = e;
				}
				
				String tag = interp.getTag(morfeusz);
				String pos = tagset.getPos(tag);
				
				if ((pos.equals("subst") || 
					pos.equals("ger") || 
					pos.equals("adj") || 
					pos.equals("ppas") || 
					pos.equals("pact")) && tag.contains("nom"))
				{
					String lemma = interp.getLemma();
					String[] str = lemma.split(":");
					
					if (str.length > 0) lemma = str[0];
					if (!list.contains(lemma)) list.add(lemma);
				}
			}
			else if (s > start) break;
		}
		if (!list.isEmpty()) lemmas.addAll(combine(list, getLemmas(end, interps)));
		return lemmas;
	}
	
	public LinkedList<String> combine(LinkedList<String> list1, LinkedList<String> list2)
	{
		if (list2.isEmpty()) return list1;
		if (list1.isEmpty()) return list2;
		
		LinkedList<String> result = new LinkedList<String>();
		
		for (String s1 : list1) {
			for (String s2 : list2) result.add(s1 + s2);
		}
		return result;
	}
	
	public LinkedList<TermReplacement> findRelatedTerms(TermEx term, String[] words, LinkedList<WordReplacement>[] substList)
	{
		LinkedList<TermReplacement> related = findRelatedTerms(0, substList.length, substList);
		
		if (related != null) related = filterTerms(term, related);
		return related;
	}
	
	public LinkedList<TermReplacement> findRelatedTerms(int i, int n, LinkedList<WordReplacement>[] substList)
	{
		LinkedList<TermReplacement> list1 = new LinkedList<TermReplacement>();
		
		for (WordReplacement r : substList[i]) {
			String[] expr = r.expression.split(" ");
			HashSet<TermEx> set = new HashSet<TermEx>();
			
			for (String w : expr) {
				LinkedList<TermEx> l = termIndex.get(w);
				
				if (l != null) {
					if (set.isEmpty()) set.addAll(l);
					else {
						set.retainAll(l);
						if (set.isEmpty()) break;
					}
				}
			}
			if (!set.isEmpty()) {
				LinkedList<WordReplacement> wr = new LinkedList<WordReplacement>();
				TermReplacement tr;
				
				wr.add(r);
				tr = new TermReplacement(set, wr);
				list1.add(tr);
			}
		}
		if (list1.isEmpty()) return null;
		if (i < n - 1) {
			LinkedList<TermReplacement> list2 = findRelatedTerms(i + 1, n, substList);
			
			if (list2 != null) {
				LinkedList<TermReplacement> related = new LinkedList<TermReplacement>();
				
				for (TermReplacement tr1 : list1) {
					for (TermReplacement tr2 : list2) {
						HashSet<TermEx> set = new HashSet<TermEx>(tr1.terms);
						
						set.retainAll(tr2.terms);
						if (!set.isEmpty()) {
							LinkedList<WordReplacement> wr = new LinkedList<WordReplacement>(tr1.replacements);
							TermReplacement tr;
							
							wr.addAll(tr2.replacements);
							tr = new TermReplacement(set, wr);
							related.add(tr);
						}
					}
				}
				if (related.isEmpty()) return null;
				return related;
			}
			else return null;
		}
		return list1;
	}
	
	public LinkedList<String> cartesianProduct(LinkedList<LinkedList<String>> sets)
	{
		if (sets.isEmpty()) return null;
		if (sets.size() == 1) return sets.getFirst();
		else {
			LinkedList<String> first = sets.removeFirst();
			LinkedList<String> product = new LinkedList<String>();
			LinkedList<String> cp = cartesianProduct(sets);
			
			for (String s1 : first) 
				for (String s2 : cp) product.add(s1 + " " + s2);
			return product;
		}
	}
	
	public LinkedList<TermReplacement> filterTerms(TermEx term, LinkedList<TermReplacement> relatedTerms)
	{
		ListIterator<TermReplacement> it = relatedTerms.listIterator();
		
		while (it.hasNext()) {
			TermReplacement tr = it.next();
			HashSet<TermEx> s = new HashSet<TermEx>();
			
			for (TermEx t : tr.terms) {
				boolean remove = false;
				
				if (t == term) remove = true;
				else if (tr.length() != t.len) remove = true;
				else if (term.len > 1) {
					LinkedList<String> words1 = new LinkedList<String>(Arrays.asList(t.str.split(" ")));
					List<String> words2 = tr.words();
					
					for (String w : words2) words1.remove(w);
					if (!words1.isEmpty()) remove = true;
					else if (!compatible(term, t, tr.replacements)) remove = true;
				}
				if (!remove) s.add(t);
			}
			if (s.isEmpty()) it.remove();
			else tr.terms = s;
		}
		if (relatedTerms.isEmpty()) return null;
		return relatedTerms;
	}
	
	public boolean compatible(TermEx term1, TermEx term2, LinkedList<WordReplacement> replacements)
	{
		TermEx cntx1, cntx2;
		String t1, t2;
		Pair<String,String[]> p1 = getHeadPhrase(term1);
		Pair<String,String[]> p2 = getHeadPhrase(term2);
		String[] ph;
		String[] hp1 = p1.second;
		String[] hp2 = p2.second;
		String[] rhp, hc;
		int n;
		boolean same = false;
		
		if (hp1 == null || hp2 == null) return false;
		ph = term1.str.split(" ");
		t1 = p1.first;
		t2 = p2.first;
		if (t1 == null || t2 == null) return false;
		t1 = getWordReplacement(t1, ph, replacements);
		if (t1 == null) return false;
		if (t1.equals(t2)) same = true;
		else if (t1.contains(" ")) {
			same = contains(t1.split(" "), t2.split(" "));
		}
		else same = false;
		if (same) {
			if (hp1.length == hp2.length) {
				rhp = WordReplacement.words(getReplacementsForNested(ph, hp1, replacements)).toArray(new String[0]);
				if (same(rhp, hp2)) {
					cntx1 = getContextTerm(term1);
					cntx2 = getContextTerm(term2);
					
					if (cntx1 == null && cntx2 == null) return true;
					if (cntx1 == null || cntx2 == null) return false;
					replacements = getReplacementsForNested(ph, cntx1.str.split(" "), replacements);
					return compatible(cntx1, cntx2, replacements);
				}
			}
			else {
				rhp = WordReplacement.words(getReplacementsForNested(ph, hp1, replacements)).toArray(new String[0]);
				if (hp1.length < hp2.length) {
					cntx1 = term1;
					while (contains(hp2, rhp)) {
						cntx1 = getContextTerm(cntx1);
						if (cntx1 == null) return false;
						hc = getHeadPhrase(cntx1).second;
						if (hc == null) return false;
						hc = WordReplacement.words(getReplacementsForNested(ph, hc, replacements)).toArray(new String[0]);
						n = rhp.length;
						rhp = Arrays.copyOf(rhp, n + hc.length);
						for (int i = 0; i < hc.length; i++) rhp[i + n] = hc[i];
						if (same(hp2, rhp)) {
							same = true;
							break;
						}
					}
					if (same) {
						cntx1 = getContextTerm(cntx1);
						cntx2 = getContextTerm(term2);
						
						if (cntx1 == null && cntx2 == null) return true;
						if (cntx1 == null || cntx2 == null) return false;
						replacements = getReplacementsForNested(ph, cntx1.str.split(" "), replacements);
						return compatible(cntx1, cntx2, replacements);
					}
				}
				else {
					cntx2 = term2;
					while (contains(rhp, hp2)) {
						cntx2 = getContextTerm(cntx2);
						if (cntx2 == null) return false;
						hc = getHeadPhrase(cntx2).second;
						if (hc == null) return false;
						n = hp2.length;
						hp2 = Arrays.copyOf(hp2, n + hc.length);
						for (int i = 0; i < hc.length; i++) hp2[i + n] = hc[i];
						if (same(hp2, rhp)) {
							same = true;
							break;
						}
					}
					if (same) {
						cntx1 = getContextTerm(term1);
						cntx2 = getContextTerm(cntx2);
						
						if (cntx1 == null && cntx2 == null) return true;
						if (cntx1 == null || cntx2 == null) return false;
						replacements = getReplacementsForNested(ph, cntx1.str.split(" "), replacements);
						return compatible(cntx1, cntx2, replacements);
					}
				}
			}
		}
		return false;
	}
	
	public String getWordReplacement(String word, String[] phrase, LinkedList<WordReplacement> replacements)
	{
		if (replacements != null) {
			ListIterator<WordReplacement> it = replacements.listIterator();
			
			for (String w : phrase) {
				String r = it.next().expression;
				
				if (w.equals(word)) return r;
			}
		}
		return null;
	}
	
	public LinkedList<WordReplacement> getReplacementsForNested(String[] term, String[] nested, LinkedList<WordReplacement> replacements)
	{
		LinkedList<WordReplacement> list = null;
		int i, j = 0;
		
		for (i = 0; i < term.length; i++) {
			for (j = 0; j < nested.length; j++) {
				if (!term[i + j].equals(nested[j])) break;
			}
			if (j == nested.length) break;
		}
		if (j == nested.length) {
			list = new LinkedList<WordReplacement>();
			j = 0;
			for (WordReplacement wr : replacements) {
				if (j >= i) list.add(wr);
				if (list.size() == nested.length) break;
				j++;
			}
		}
		return list;
	}
	
	public void associate(TermEx term, LinkedList<TermReplacement> relatedTerms)
	{
		for (TermReplacement tr : relatedTerms) {
			boolean synonym = true;
			
			if (term.len > 1) {
				for (WordReplacement wr : tr.replacements) {
					if (wr.relTypes != null) {
						for (Integer relType : wr.relTypes) {
							if (!WordNet.isSynonymic(relType)) synonym = false;
						}
					}
				}
			}
			else synonym = false;
			for (TermEx t : tr.terms) {
				if (synonym) {
					if (!areSynonyms(t, term)) term.addEquivalentTerm(t, termMap);
				}
				else if (!areRelated(t, term)) term.relateWith(t, tr.replacements);
			}
		}
	}
	
	public boolean areSynonyms(TermEx term1, TermEx term2)
	{
		if (term1 == term2) return true;
		
		LinkedList<String> children = term2.getChildren();
		
		if (children != null) {
			for (String c : children) {
				TermEx t = (TermEx)termMap.get(c);
				
				if (areRelated(term1, t)) return true;
			}
		}
		return false;
	}
	
	public boolean areRelated(TermEx term1, TermEx term2)
	{
		if (term1 == term2) return true;
		
		LinkedList<String> children = term2.getChildren();
		
		if (children != null) {
			for (String c : children) {
				TermEx t = (TermEx)termMap.get(c);
				
				if (areRelated(term1, t)) return true;
			}
		}
		
		LinkedList<String> equivalent = term2.getEquivalentTerms();
		
		if (equivalent != null) {
			if (equivalent.contains(term1.id)) return true;
		}
		
		if (TermEx.containedInRelated(term1, term2.getRelatedTerms())) return true;
		
		return false;
	}
	
	public void addNewResults(File[] newFiles)
	{
		Finalizer finalizer = new Finalizer();
		
		modified = true;
		cancelled = false;
		interrupted = false;
		rebuild = 2;
		extractor = new ExtractorEngine(this, newFiles);
		formsCollected = false;
		sentencesIndexed = false;
		System.gc();
		
		GroupingEngine groupingEngine = null;
		
		if (preferences.makeGroups) groupingEngine = new GroupingEngine();
		
		ExtractorThread et = new ExtractorThread(extractor, groupingEngine, finalizer, null);
		
		et.start();
		if (TermoPL.batchMode) {
			try {
				et.join();
			}
			catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
	}
	
	public void calculate(int cntxMethod)
	{
		for (Term term : terms) term.calc(cntxMethod);
	}
	
	public HashMap<String, Term> loadContrastiveTerms(File file)
	{
		HashMap<String, Term> newCTerms = null;
		
		if (file.getName().toLowerCase().endsWith(".trm")) {
			ObjectInputStream is;
			
			try {
				is = new ObjectInputStream(new FileInputStream(file));
				newCTerms = loadContrastiveTerms(is);
				is.close();
			}
			catch (Exception exception) {
				if (TermoPL.batchMode) {
					System.out.println("Error occured while reading contrastive data.");
				}
				else {
					JOptionPane.showMessageDialog(TermoPL.dialogOwner, "Error occured while reading contrastive data.", "Error", JOptionPane.ERROR_MESSAGE);
				}
				newCTerms = null;
			}
		}
		else {
			BufferedReader reader;
			
			try {
				reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF8"));
				newCTerms = loadContrastiveTerms(reader);
				reader.close();
			}
			catch (IOException exception) {
				newCTerms = null;
			}
		}
		return newCTerms;
	}
	
	public HashMap<String, Term> loadContrastiveTerms(BufferedReader reader) throws IOException
	{
		String line;
		HashMap<String, Term> newCTerms = new HashMap<String, Term>();
		boolean done = false;
		
		do {
			line = reader.readLine();
			if (line == null) done = true;
			else {
				line = line.trim().replace(",", ".");
				if (line.isEmpty()) done = true;
				else {	
					String[] s = line.split("\t");
					Term t = new Term();
					
					try {
						if (s.length == 3) {
							t.str = s[0];
							t.cvalue = Double.parseDouble(s[1]);
							t.freq_s = Integer.parseInt(s[2]);
							t.contrast = 0.0;
						}
						else if (s.length == 8) {
							Integer.parseInt(s[0]);
							t.rank = Integer.parseInt(s[1]);
							t.str = s[2];
							t.cvalue = Double.parseDouble(s[3]);
							t.contrast = 0.0;
							t.len = Integer.parseInt(s[4]);
							t.freq_s = Integer.parseInt(s[5]);
							t.freq_in = Integer.parseInt(s[6]);
							t.lk = Integer.parseInt(s[7]);
						}
						else if (s.length == 9) {
							Integer.parseInt(s[0]);
							t.rank = Integer.parseInt(s[1]);
							t.str = s[2];
							if (s[3].startsWith("[")) {
								t.cvalue = Double.parseDouble(s[4]);
								t.contrast = 0.0;
							}
							else {
								t.cvalue = Double.parseDouble(s[3]);
								t.contrast = 0.0;
							}
							t.len = Integer.parseInt(s[5]);
							t.freq_s = Integer.parseInt(s[6]);
							t.freq_in = Integer.parseInt(s[7]);
							t.lk = Integer.parseInt(s[8]);
						}
						else if (s.length == 10) {
							Integer.parseInt(s[0]);
							t.rank = Integer.parseInt(s[1]);
							t.str = s[2];
							t.cvalue = Double.parseDouble(s[4]);
							t.contrast = 0.0;
							t.len = Integer.parseInt(s[6]);
							t.freq_s = Integer.parseInt(s[7]);
							t.freq_in = Integer.parseInt(s[8]);
							t.lk = Integer.parseInt(s[9]);
						}
						else break;
						newCTerms.put(t.str, t);
					}
					catch (Exception exception) {
						if (TermoPL.batchMode) {
							System.out.println("Error occured while reading contrastive data.");
						}
						else {
							JOptionPane.showMessageDialog(TermoPL.dialogOwner, "Error occured while reading contrastive data.", "Error", JOptionPane.ERROR_MESSAGE);
						}
						break;
					}
				}
			}
		} while (!done && !cancelled);
		if (!done || cancelled) newCTerms = null;
		return newCTerms;
	}
	
	public HashMap<String, Term> loadContrastiveTerms(ObjectInputStream is) throws Exception
	{
		int nterms;
		HashMap<String, Term> newCTerms;
		
		is.readBoolean();
		is.readBoolean();
		is.readInt();
		is.readInt();
		nterms = is.readInt();
		is.readInt();
		is.readInt();
		is.readObject();
		is.readObject();
		is.readObject();
		is.readObject();
		is.readObject();
		is.readObject();
		newCTerms = new HashMap<String, Term>(nterms);
		for (int i = 0; i < nterms && !cancelled; i++) {
			Term t = (Term)is.readObject();
			String str = calcSimplifiedForm(t);
			
			newCTerms.put(str, t);
		}
		if (cancelled) newCTerms = null;
		is.close();
		return newCTerms;
	}
	
	public void cancel()
	{
		cancelled = true;
		if (extractor != null) extractor.cancel();
		if (loaded) {
			if (termsView != null) {
				termsView.showProgress(false);
				termsView.setWarning(TermsView.CANCEL_WARNING);
				termsView.setAccessory(extractAction);
			}
			if (formsView != null) formsView.reset();
			if (sentencesView != null) sentencesView.reset();
			if (groupsView != null) groupsView.reset();
		}
	}
	
	public void interrupt()
	{
		interrupted = true;
		if (extractor != null) extractor.cancel();
		if (loaded) {
			if (termsView != null) {
				termsView.showProgress(false);
			}
			if (formsView != null) formsView.reset();
			if (sentencesView != null) sentencesView.reset();
			if (groupsView != null) groupsView.reset();
		}
	}
	
	public void trim(boolean trimResults, boolean multiWordTermsOnly, boolean filterResults) 
	{
		if (preferences.trimResults != trimResults) {
			TermoPL.preferences.trimResults = preferences.trimResults = trimResults;
			TermoPL.preferences.setModified(true);
		}
		if (preferences.multiWordTermsOnly != multiWordTermsOnly) {
			TermoPL.preferences.multiWordTermsOnly = preferences.multiWordTermsOnly = multiWordTermsOnly;
			TermoPL.preferences.setModified(true);
		}
		if (preferences.filterResults != filterResults) {
			TermoPL.preferences.filterResults = preferences.filterResults = filterResults;
			TermoPL.preferences.setModified(true);
		}
		
		if (terms != null) {
			int tsize = terms.length;
			String str = (termsView != null ? termsView.getSearchString() : "");
			
			if (filterResults && !str.equals("")) tsize = trimMismatched(tsize);
			if (multiWordTermsOnly) tsize = trimTermsByLength(tsize);
			if (trimResults) tsize = trimLowRanked(tsize);
			tableSize = tsize;
			sortTable(0, tableSize);
			if (termsView != null) {
				termsView.resetResults();
				termsView.resetSearch();
			}
			System.gc();
		}
	}
	
	public int trimMismatched(int tsize)
	{
		int i, j;
		boolean found;
		Term x;
		
		i = 0;
		while (i < tsize) {
			if (!terms[i].isMatching()) {
				j = i + 1;
				found = false;
				while (j < tsize) {
					if (terms[j].isMatching()) {
						x = terms[j];
						terms[j] = terms[i];
						terms[i] = x;
						found = true;
						break;
					}
					else j++;
				}
				if (!found) break;
			}
			i++;
		}
		tsize = i;
		return tsize;
	}
	
	public int trimTermsByLength(int tsize)
	{
		int i, j;
		boolean found;
		Term x;
		
		i = 0; 
		while (i < tsize) {
			if (terms[i].len < preferences.minLength || (preferences.maxLength > 0 && terms[i].len > preferences.maxLength)) {
				j = i + 1;
				found = false;
				while (j < tsize) {
					if (terms[j].len >= preferences.minLength && (preferences.maxLength < 0 || terms[j].len <= preferences.maxLength)) {
						x = terms[j];
						terms[j] = terms[i];
						terms[i] = x;
						found = true;
						break;
					}
					else j++;
				}
				if (!found) break;
			}
			i++;
		}
		tsize = i;
		return tsize;
	}
	
	public int trimLowRanked(int tsize)
	{
		int i, m, n, r, sc, sp;
		
		sc = preferences.sortedColumn;
		preferences.sortedColumn = TermoPLConsts.RANK;
		sp = preferences.sortPrefs[TermoPLConsts.RANK];
		preferences.sortPrefs[TermoPLConsts.RANK] = 1;
		Arrays.sort(terms, 0, tsize, sorter);
		i = m = n = r = 0;
		for (i = 0; i < tsize; i++) {
			Term t = terms[i];
			
			if (t.rank > r) {
				if (m >= preferences.maxResults) break;
				r = t.rank;
				m++;
			}
			n++;
		}
		tsize = n;
		preferences.sortPrefs[TermoPLConsts.RANK] = sp;
		preferences.sortedColumn = sc;
		return tsize;
	}
	
	public void selectFiles()
	{
		addNewSourceFiles();
		switch (rebuild) {
			case 0 :
				if (termsView != null) {
					termsView.setInfo(getStat());					
				}
				break;
			case 1 :
			case 2 :
				if (newFiles != null) {
					preferences.filePath = newFiles[0].getParentFile().getPath();
					TermoPL.preferences.filePath = preferences.filePath;
					TermoPL.preferences.setModified(true);
				}
				if (termsView != null) {
					termsView.setWarning(TermsView.EXTRACT_WARNING);
					termsView.setAccessory(extractAction);
				}
				break;
			case 3 :
				if (termsView != null) {
					termsView.setWarning(TermsView.SELECT_WARNING);
					termsView.setAccessory(selectAction);
				}
		}
	}
	
	public File[] selectSourceFiles()
	{
		JFileChooser chooser = new JFileChooser();
		
		chooser.setDialogTitle("Select File(s)");
		chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
		chooser.setApproveButtonText("Select");
		chooser.setApproveButtonMnemonic('S');
		chooser.setMultiSelectionEnabled(true);
		if (termsView != null) termsView.setAccessory(null);
		if (preferences.filePath != null) {
			File file = new File(preferences.filePath);
			
			if (file.exists()) chooser.setCurrentDirectory(file);
		}
		if (chooser.showOpenDialog(TermoPL.dialogOwner) == JFileChooser.APPROVE_OPTION)
			return TermoPL.collectFiles(chooser.getSelectedFiles());
		return null;
	}
	
	public void addNewSourceFiles()
	{
		SelectedFiles sf = new SelectedFiles(this);
		int res = sf.doDialog();
		
		if (res == SelectedFiles.OK) {
			corpusName = sf.getCorpusName();
			
			newFiles = sf.getNewFiles();
			oldFiles = sf.getOldFiles();
			rebuild = sf.rebuild();
			reuseTaggedFiles = sf.reuse();
			if (preferences.reuseTaggedFiles != reuseTaggedFiles) {
				preferences.reuseTaggedFiles = reuseTaggedFiles;
				if (!reuseTaggedFiles) {
					preferences.makeIndex = false;
					TermoPL.preferences.makeIndex = false;
				}
				TermoPL.preferences.reuseTaggedFiles = reuseTaggedFiles;
				TermoPL.preferences.save();
			}
		}
		else if (selectedFiles == null) rebuild = 3;
		sf.dispose();
	}
	
	public void selectContrastiveTerms()
	{
		String path = TermoPL.selectContrastiveCorpus();
		if (path != null) {
			preferences.contrastiveDataPath = path;
			if (termsView != null && preferences.applyContrastiveRanking) {
				termsView.setInfo(TermsView.COMPARE_WARNING);
				termsView.setAccessory(compareAction);
			}
		}
	}
	
	public void compare()
	{
		boolean load = false;
		ContrastiveDataLoader loader = null;
		
		cancelled = false;
		preferences.applyContrastiveRanking = true;
		if (cterms == null) {
			if (preferences.contrastiveDataPath == null) selectContrastiveTerms();
			if (preferences.contrastiveDataPath != null) load = true;
		}
		else if (preferences.reloadContrastiveTerms) load = true;
		if (load) {
			if (termsView != null) {
				termsView.setAccessory(null);
				termsView.showProgress(true);
			}
			loader = new ContrastiveDataLoader(preferences.contrastiveDataPath);
		}
		
		CompareThread ct = new CompareThread(loader);
		
		ct.start();
	}
	
	public void finishCompare()
	{
		if (preferences.sortedColumn == TermoPLConsts.CSVALUE) Arrays.sort(terms, 0, tableSize, sorter);
		if (termsView != null) {
			EventQueue.invokeLater(new Runnable() {
				public void run() 
				{
					termsView.setInfo(getStat());
					termsView.setAccessory(null);
					termsView.showProgress(false);
					termsView.refreshColumns();
				}
			});
		}
		preferences.compare = false;
		preferences.reloadContrastiveTerms = false;
		preferences.repaint = false;
		modified = true;
	}
	
	public void merge()
	{
		JFileChooser chooser = new JFileChooser();
		
		chooser.setDialogTitle("Merge");
		chooser.setAcceptAllFileFilterUsed(false);
		chooser.setFileFilter(TermoPL.TRM_FILTER);
		chooser.setApproveButtonText("Merge");
		chooser.setApproveButtonMnemonic('M');
		chooser.setApproveButtonToolTipText("Merge terms from selected file");
		if (preferences.filePath != null) {
			File file = new File(preferences.filePath);
			
			if (file.exists()) chooser.setCurrentDirectory(file);
		}
		if (chooser.showOpenDialog(TermoPL.dialogOwner) == JFileChooser.APPROVE_OPTION) {
			File file = chooser.getSelectedFile();
			
			TermoPL.preferences.filePath = preferences.filePath = file.getParentFile().getPath();
			TermoPL.preferences.setModified(true);
			merge(file);
		}
	}
	
	public void merge(File file)
	{
		MergeThread mt = new MergeThread(file);
		
		if (termsView != null) {
			termsView.showProgress(true);
			termsView.changeProgress(3);
		}
		if (groupsView != null) {
			close(getWindow(GROUPS_WINDOW));			
		}
		mt.start();
		if (TermoPL.batchMode) {
			try {
				mt.join();
			}
			catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
	}
	
	public void finalizeMerge()
	{
		if (termsView != null) {
			termsView.showProgress(false);
			termsView.setInfo(getStat());
			termsView.resetResults();
		}
	}
	
	public void allowCollectingForms(boolean f)
	{
		preferences.collectAllForms = f;
		if (preferences.collectAllForms) {
			if (termsView != null) {
				termsView.setWarning(TermsView.EXTRACT_WARNING);
				termsView.setAccessory(extractAction);
			}
		}
		else {
			TermoPLWindow wnd = getWindow(FORMS_WINDOW);
			
			if (wnd != null) wnd.close();
			if (terms != null)
				for (Term t : terms) t.trimForms();
			formsCollected = false;
		}
		modified = true;
	}
	
	public void allowIndexingSentences(boolean f)
	{
		preferences.makeIndex = f;
		if (preferences.makeIndex) {
			if (termsView != null) {
				termsView.setWarning(TermsView.EXTRACT_WARNING);
				termsView.setAccessory(extractAction);
			}
		}
		else {
			TermoPLWindow wnd = getWindow(SENTENCES_WINDOW);
			
			if (wnd != null) wnd.close();
			if (terms != null)
				for (Term t : terms) t.trimIndex();
			sentencesIndexed = false;
		}
		modified = true;
	}
	
	public boolean allowClose()
	{
		if (terms != null && modified) {
			int answer = JOptionPane.showConfirmDialog(TermoPL.dialogOwner, 
							"Term set '" + getDocumentDisplayName() + "' has been modified.\n" + "Do you want to accept the changes?",
							"Term set modification",
							JOptionPane.YES_NO_CANCEL_OPTION,
							JOptionPane.WARNING_MESSAGE);
			
			if (answer == JOptionPane.YES_OPTION) {
				save();
				if (modified) return false;
				return true;
			}
			else if (answer == JOptionPane.NO_OPTION) return true;
			return false;
		}
		return true;
	}
	
	public boolean allowClose(TermoPLWindow wnd)
	{
		boolean ok = true;
		
		if (wnd.getID() == TERMS_WINDOW) ok = allowClose();
		return ok;
	}
	
	public void close()
	{
		if (allowClose()) {
			waitForSaveThreadsToFinish();
			remove();
		}
	}
	
	public void close(TermoPLWindow wnd)
	{
		if (wnd.getID() == TERMS_WINDOW) close();
		else remove(wnd);
	}
	
	public void remove()
	{
		while (!windows.isEmpty()) {
			TermoPLWindow wnd = windows.removeFirst();
			
			remove(wnd);
		}
		TermoPL.application.closeDocument(this);
		System.gc();
	}
	
	public void remove(TermoPLWindow wnd)
	{
		switch (wnd.getID()) {
			case TERMS_WINDOW:
				termsView = null;
				break;
			case FORMS_WINDOW:
				formsView = null;
				break;
			case SENTENCES_WINDOW:
				sentencesView = null;
				break;
			case GROUPS_WINDOW:
				groupsView = null;
		}
		windows.remove(wnd);
		TermoPL.zorder.remove(wnd);
		wnd.destroySelf();
	}
	
	public void save()
	{
		if (input == null || !input.getName().equals(getOutputName() + ".trm")) saveAs();
		else save(input);
	}
	
	public void saveAs()
	{
		SaveFileChooser chooser = new SaveFileChooser();
		String name = getOutputName() + ".trm";
		
		chooser.setDialogTitle("Save");
		chooser.setAcceptAllFileFilterUsed(false);
		chooser.setFileFilter(TermoPL.TRM_FILTER);
		chooser.setSelectedFile(new File(name));
		if (preferences.filePath != null) {
			File file = new File(preferences.filePath);
			
			if (file.exists()) chooser.setCurrentDirectory(file);
		}
		if (chooser.showSaveDialog(TermoPL.dialogOwner) == JFileChooser.APPROVE_OPTION) {
			File file = chooser.getSelectedFile();
			
			MenuFactory.removeFromDocumentSwitchMenu(this);
			input = file;
			name = file.getName();
			docDisplayName = TermoPL.getDisplayName(name);
			docName = name;
			TermoPL.preferences.filePath = preferences.filePath = file.getParentFile().getPath();
			TermoPL.preferences.setModified(true);
			TermoPL.application.updateRecentFiles(file.getPath());
			MenuFactory.addToDocumentSwitchMenu(this);
			changeTitles();
			save(file);
		}
	}
	
	public void save(File file)
	{
		SaveThread saveThread = new SaveThread(file);
		
		if (!TermoPL.batchMode && termsView != null) {
			termsView.setAccessory(null);
			termsView.showProgress(true, false);
			termsView.pleaseWait();
		}
		saveThreads[SAVE_THREAD] = saveThread;
		saveThread.start();
	}
	
	public void save(ObjectOutputStream output) throws IOException
	{
		int ncterms;
		
		output.writeBoolean(formsCollected);
		output.writeBoolean(sentencesIndexed);
		output.writeInt(nsentences);
		output.writeInt(ntokens);
		output.writeInt(nterms);
		if (cterms == null) ncterms = 0;
		else ncterms = cterms.size();
		output.writeInt(ncterms);
		output.writeInt(rebuild);
		output.writeObject(corpusName);
		preferences.save(output);
		if (selectedFiles == null) output.writeObject(selectedFiles);
		else {
			String[] ftable = new String[selectedFiles.length];
			
			for (int i = 0; i < selectedFiles.length; i++) {
				File f = selectedFiles[i];
				
				ftable[i] = preferences.getRelativizedPath(preferences.workSpace, f.getAbsolutePath(), "WORKSPACE");
			}
			output.writeObject(ftable);
		}
		if (newFiles == null) output.writeObject(newFiles);
		else {
			String[] ftable = new String[newFiles.length];
			
			for (int i = 0; i < newFiles.length; i++) {
				File f = newFiles[i];
				
				ftable[i] = preferences.getRelativizedPath(preferences.workSpace, f.getAbsolutePath(), "WORKSPACE");
			}
			output.writeObject(ftable);
		}
		if (oldFiles == null) output.writeObject(oldFiles);
		else {
			String[] ftable = new String[oldFiles.length];
			
			for (int i = 0; i < oldFiles.length; i++) {
				File f = oldFiles[i];
				
				ftable[i] = preferences.getRelativizedPath(preferences.workSpace, f.getAbsolutePath(), "WORKSPACE");
			}
			output.writeObject(ftable);
		}
		if (analyzedFiles == null) output.writeObject(null);
		else {
			ArrayList<Pair<String, Integer>> ftable = new ArrayList<Pair<String, Integer>>(analyzedFiles.size());
			
			for (FileDescr fd : analyzedFiles) {
				ftable.add(new Pair<String, Integer>(
					preferences.getRelativizedPath(preferences.workSpace, fd.file.getAbsolutePath(), "WORKSPACE"), fd.type));
			}
			try {
				output.writeObject(ftable);
			}
			catch (Exception exception) {
				exception.printStackTrace();
			}
		}
		
		if (terms != null) {
			for (Term t : terms) output.writeObject(t);
		}
		if (cterms != null) {
			for (Map.Entry<String, Term> entry : cterms.entrySet()) {
				output.writeObject(entry.getKey());
				output.writeObject(entry.getValue());
			}
		}
		output.flush();
		output.close();
		modified = false;
	}
	
	public void selectWorkspace()
	{
		Workspace workspace = new Workspace(preferences.workSpace);
		int res = workspace.doDialog();
		
		if (res == Workspace.OK) {
			String newWorkSpace = workspace.getCurrentWorkspace();
			
			if (!newWorkSpace.equals(preferences.workSpace)) {
				preferences.workSpace = preferences.filePath = workspace.getCurrentWorkspace();
				TermoPL.preferences.workSpace = TermoPL.preferences.filePath = preferences.workSpace;
				TermoPL.preferences.setModified(true);
				modified = true;
			}
		}
		workspace.dispose();
	}
	
	public void export(int exportType)
	{
		SaveFileChooser chooser = new SaveFileChooser();
		String name = getOutputName();
		
		switch (exportType) {
			case EXPORT_RESULTS:
				name += ".txt";
				chooser.setDialogTitle("Export Results");
				break;
			case EXPORT_FORMS:
				name += "_forms.txt";
				chooser.setDialogTitle("Export Forms");
				break;
			case EXPORT_SENTENCES:
				name += "_sentences.txt";
				chooser.setDialogTitle("Export Sentences");
				break;
			case EXPORT_GROUPS:
				name += "_groups.xml";
				chooser.setDialogTitle("Export Term Groups");
		}
		chooser.setSelectedFile(new File(name));
		chooser.setApproveButtonText("Export");
		chooser.setApproveButtonMnemonic('E');
		if (preferences.filePath != null) {
			File file = new File(preferences.filePath);
			
			if (file.exists()) chooser.setCurrentDirectory(file);
		}
		if (chooser.showSaveDialog(TermoPL.dialogOwner) == JFileChooser.APPROVE_OPTION) {
			File file = chooser.getSelectedFile();
			
			TermoPL.preferences.filePath = preferences.filePath = file.getParentFile().getPath();
			TermoPL.preferences.setModified(true);
			export(file, exportType);
		}
	}
	
	public void export(File file, int exportType)
	{
		try {
			PrintWriter pw = new PrintWriter(file, "UTF8");
			ExportThread exportThread = new ExportThread(pw, exportType);
				
			if (!TermoPL.batchMode && termsView != null) {
				termsView.setAccessory(null);
				termsView.showProgress(true, false);
				termsView.pleaseWait();
			}
			switch (exportType) {
				case EXPORT_RESULTS:
					saveThreads[EXPORT_RESULTS_THREAD] = exportThread;
					break;
				case EXPORT_FORMS:
					saveThreads[EXPORT_FORMS_THREAD] = exportThread;
					break;
				case EXPORT_SENTENCES:
					saveThreads[EXPORT_SENTENCES_THREAD] = exportThread;
					break;
				case EXPORT_GROUPS:
					saveThreads[EXPORT_GROUPS_THREAD] = exportThread;
			}
			exportThread.start();
		} 
		catch (FileNotFoundException e) {
			e.printStackTrace();
		} 
		catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
	}
	
	public void export(PrintWriter pw, int exportType)
	{
		switch (exportType) {
			case EXPORT_RESULTS:
				exportResults(pw);
				break;
			case EXPORT_FORMS:
				exportForms(pw);
				break;
			case EXPORT_SENTENCES:
				exportSentences(pw);
				break;
			case EXPORT_GROUPS:
				exportGroups(pw);
				break;
		}
		pw.flush();
		pw.close();
	}
	
	public void exportResults(PrintWriter pw)
	{
		BaseFormGuesser guesser = null;
		
		if (preferences.saveBF && !preferences.calculateBaseForms)
			if (preferences.useUD) guesser = new BaseFormGuesser();
			else guesser = new BaseFormGuesser(preferences.tagset);
		
		for (int i = 0; i < tableSize; i++) {
			Term t = terms[i];
			boolean addTab = false;
			
			if (preferences.saveCount) {
				pw.print(i + 1);
				addTab = true;
			}
			if (preferences.saveRank) {
				if (addTab) pw.print("\t");
				pw.print(t.rank);
				addTab = true;
			}
			if (preferences.saveSF) {
				if (addTab) pw.print("\t");
				if (preferences.calculateBaseForms)	pw.print(calcSimplifiedForm(t));
				else pw.print(t.str);
				addTab = true;
			}
			if (preferences.saveBF) {
				if (addTab) pw.print("\t");
				if (preferences.saveSF) pw.print("[");
				if (preferences.calculateBaseForms) pw.print(t.str);
				else pw.print(guesser.calcBaseForm(t));
				if (preferences.saveSF) pw.print("]");
				addTab = true;
			}
			if (preferences.saveCV) {
				if (addTab) pw.print("\t");
				pw.print(t.cvalue);
				addTab = true;
			}
			if (preferences.saveComp) {
				if (preferences.applyContrastiveRanking && cterms != null) {
					if (addTab) pw.print("\t");
					pw.print(t.contrast);
					addTab = true;
				}
			}
			if (preferences.saveLen) {
				if (addTab) pw.print("\t");
				pw.print(t.len);
				addTab = true;
			}
			if (preferences.saveFreqs) {
				if (addTab) pw.print("\t");
				pw.print(t.freq_s);
				addTab = true;
			}
			if (preferences.saveFreqin) {
				if (addTab) pw.print("\t");
				pw.print(t.freq_in);
				addTab = true;
			}
			if (preferences.saveContext) {
				if (addTab) pw.print("\t");
				pw.print(t.lk);
			}
			pw.println();
		}
	}
	
	public void exportForms(PrintWriter pw)
	{
		for (int i = 0; i < tableSize; i++) {
			Term t = terms[i];
			boolean appendSep = false;
			
			pw.print("[");
			pw.print(t.str);
			pw.println("]");
			pw.println();
			if (t.getForms() != null) {
				for (Form f : t.getForms()) {
					String str = f.toString();
					
					if (appendSep) pw.print(", ");
					pw.print(str);
					appendSep = true;
				}
				pw.println(); pw.println();
			}
		}
	}
	
	public void exportSentences(PrintWriter pw)
	{
		RandomAccessFile source = null;
		File sourceFile = null;
		boolean appendSpace;
		
		for (int i = 0; i < tableSize; i++) {
			Term t = terms[i];
			
			pw.print("[");
			pw.print(t.str);
			pw.println("]");
			pw.println();
			
			long pos = -1, oldPos = -1;
			int fileID = -1, oldFileID = -1;
			
			if (t.getSentenceRef() != null) {
				for (SentenceRef ref : t.getSentenceRef()) {
					if (ref instanceof SentenceRefEx) {
						fileID = ((SentenceRefEx) ref).fileID;
					}
					pos = ref.start;
					
					if (pos != oldPos || fileID != oldFileID) {
						Pair<LinkedList<Token>, LinkedList<MultiWordToken>> pair;
						LinkedList<Token> sentence = null;
						LinkedList<MultiWordToken> mwtList = null;
						FileDescr fd = getAnalyzedFile(ref.getFileID());
						
						oldPos = pos;
						oldFileID = fileID;
						try {
							if (sourceFile != fd.file) {
								sourceFile = fd.file;
								source = new RandomAccessFile(sourceFile, "r");
							}
							pair = CorpusReader.getSentence(source,  fd.type, ref.start, ref.len);
							sentence = pair.first;
							mwtList = pair.second;
							if (mwtList != null) sentence = CorpusReader.replaceMWT(sentence, mwtList);
							appendSpace = false;
							for (Token tok : sentence) {
								if (appendSpace) pw.print(" ");
								pw.print(tok.form);
								appendSpace = tok.spaceAfter;
							}
							pw.println();
						}
						catch (Exception e) {
							e.printStackTrace();
							System.exit(0);
						}
					}
				}
				pw.println();
			}
		}
	}
	
	public void exportGroups(PrintWriter pw)
	{
		pw.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
		pw.write("<term_list>\n");
		for (Term term : terms) {
			TermEx t = (TermEx)term;
			
			exportTerm(t, pw);
		}
		pw.write("</term-list>\n");
	}
	
	public void exportTerm(TermEx term, PrintWriter pw)
	{
		pw.write("\t<term id=\"" + toXML(term.id) + "\" name=\"" + toXML(term.str) + "\">\n");
		exportEquivalentTerms(term, pw);
		exportTerms(term.collectParents(termMap), "parent", pw);
		exportTerms(term.collectChildren(termMap), "child", pw);
		exportRelated(term, pw);
		pw.write("\t</term>\n");
	}
	
	public void exportEquivalentTerms(TermEx term, PrintWriter pw)
	{
		LinkedList<String> equiv = term.getEquivalentTerms();
		
		if (equiv != null) {
			for (String e : equiv) 
				if (e != term.id) pw.write("\t\t<eq id=\"" + e + "\"/>\n");
		}
	}
	
	public void exportTerms(LinkedList<String> terms, String elementName, PrintWriter pw)
	{
		if (terms != null) {
			for (String t : terms) pw.write("\t\t<" + elementName + " id=\"" + toXML(t) + "\"/>\n");
		}
	}
	
	public void exportRelated(TermEx term, PrintWriter pw)
	{
		LinkedList<Pair<String, LinkedList<WordReplacement>>> related = term.collectRelated(termMap);
		
		if (related != null) {
			for (Pair<String, LinkedList<WordReplacement>> p : related) {
				pw.write("\t\t<rel id=\"" + p.first + "\">\n");
				for (WordReplacement wr : p.second) {
					LinkedList<Integer> relTypes = wr.relTypes;
					
					if (relTypes == null) pw.write("\t\t\t<nosubst/>\n");
					else {
						String repl = (relTypes.getFirst() < 0 ? wr.expression : wr.word);

						pw.write("\t\t\t<subst word=\"" + toXML(wr.word) + "\" expr=\"" +
							toXML(wr.expression) + "\" replaced=\"" + toXML(repl) + "\">\n");
						for (Integer type : relTypes) {
							if (type < 0) type = -type;
							pw.write("\t\t\t\t<reltype id=\"" + type + "\" name=\"" + toXML(WordNet.getName(type)) + "\"/>\n");
						}
						pw.write("\t\t\t</subst>\n");
					}
				}
				pw.write("\t\t</rel>\n");
			}
		}
	}
	
	public String toXML(String str)
	{
		StringBuffer buffer = new StringBuffer();
		
		for (int i = 0; i < str.length(); i++) {
			Character ch = str.charAt(i);
			
			if (ch == '&') buffer.append("&amp;");
			else if (ch == '<') buffer.append("&lt;");
			else if (ch == '>') buffer.append("&gt;");
			else if (ch == '\'') buffer.append("&apos;");
			else if (ch == '"') buffer.append("&quot;");
			else buffer.append(ch);
		}
		return buffer.toString();
	}
	
	public String getOutputName()
	{
		String contrastiveCorpusName = null;
		
		if (cterms != null) contrastiveCorpusName = TermoPL.getFileName(preferences.contrastiveDataPath);
		return getOutputName(corpusName, contrastiveCorpusName);
	}
	
	public String getOutputName(String corpusName, String contrastiveCorpusName)
	{
		String name = "";
		
		if (preferences.applyContrastiveRanking && cterms != null) name = "comp_";
		if (preferences.trimResults) name += "top_ranked_";
		if (preferences.multiWordTermsOnly) name += "multi-word_terms_";
		name += corpusName;
		if (preferences.useUD) name += "_UD";
		else if (preferences.useNPMIMethod) name += "_NPMI" + preferences.NPMIMethod;
		else name += "_NONPMI";
		if (preferences.applyContrastiveRanking && cterms != null) {
			name += "_vs_" + contrastiveCorpusName + "_";
			switch (preferences.contrastiveRankingMethod) {
				case 1 : name += "LL"; break;
				case 2 : name += "TFITF"; break;
				case 3 : name += "CSmw"; break;
				case 4 : name += "TW";
			}
		}
		return name;
	}
	
	public void createWindow(int ID)
	{
		TermoPLWindow wnd;
		Container view;
		String title;
		
		switch (ID) {
			case TERMS_WINDOW:
				view = termsView = new TermsView(this);
				title = "Terms";
				break;
			case FORMS_WINDOW:
				view = formsView = new FormsView(this);
				title = "Forms";
				break;
			case SENTENCES_WINDOW:
				view = sentencesView = new SentencesView(this);
				title = "Sentences";
				break;
			case GROUPS_WINDOW:
				view = groupsView = new GroupsView(this);
				title = "Related Term Groups";
				break;
			default:
				title = "";
				view = null;
		}
		if (TermoPL.isMacOS) wnd = new TermoPLMacWindow(this, ID, view);
		else wnd = new TermoPLDefaultWindow(this, ID, view);
		wnd.setTitle(title + " [" + getDocumentDisplayName() + "]");
		windows.add(wnd);
		TermoPL.zorder.add(wnd);
		setWindowBounds(wnd);
		((Component)wnd).setVisible(true);
	}
	
	public void setWindowBounds(TermoPLWindow wnd)
	{
		Rectangle rect = preferences.getBounds(wnd.getID());
		
		if (rect.width == 0) {
			Rectangle r;
			
			if (TermoPL.isMacOS) r = WindowUtils.getScreenRect();
			else r = WindowUtils.getDesktopRect();
			switch (wnd.getID()) {
				case TERMS_WINDOW:
					rect.width = wnd.getPreferredSize().width;
					rect.height = (int)(0.9 * r.getHeight());
					break;
				case FORMS_WINDOW:
					rect.width = 400;
					rect.height = 200;
					break;
				case SENTENCES_WINDOW:
				case GROUPS_WINDOW:
					rect.width = (int)(0.5 * r.getWidth());
					rect.height = (int)(0.9 * r.getHeight());
			}
			wnd.setBounds(rect);
			if (TermoPL.isMacOS) WindowUtils.centerOnScreen((Component)wnd);
			else WindowUtils.centerOnDesktop((Component)wnd);
		}
		else wnd.setBounds(rect);
	}
	
	public TermoPLWindow getWindow(int ID)
	{
		for (TermoPLWindow wnd : windows) {
			if (wnd.getID() == ID) return wnd;
		}
		return null;	
	}
	
	public LinkedList<TermoPLWindow> getWindows()
	{
		return windows;
	}
	
	public void changeTitles()
	{
		for (TermoPLWindow wnd : windows) {
			String title = wnd.getTitle();
			int i = title.indexOf('[');
			
			title = title.substring(0, i) + "[" + getDocumentDisplayName() + "]";
			wnd.setTitle(title);
		}
	}
	
	public String getStat()
	{
		String stat = "<html><b>Corpus name:</b> " + corpusName + "; " +
					  "<b>Sentences:</b> " + nsentences + "; " +
					  "<b>Tokens:</b> " + ntokens + "; " +
					  "<b>Terms:</b> " + nterms;
		
		if (preferences.applyContrastiveRanking && cterms != null) 
			stat += "; <b>Compared with:</b> " + TermoPL.getFileName(preferences.contrastiveDataPath);
		stat += "</html>";
		return stat;
	}
	
	public void prepareTable()
	{
		int rank = 1;
		int sc = preferences.sortedColumn;
		int s = preferences.sortPrefs[TermoPLConsts.CVALUE];
		double lastValue = -1.0;
		
		if (!TermoPL.batchMode && termsView != null) termsView.finalize();
		preferences.sortedColumn = TermoPLConsts.CVALUE;
		preferences.sortPrefs[TermoPLConsts.CVALUE] = -1;
		sortTable();
		for (Term term : terms) {
			term.rank = rank;
			if (lastValue >= 0.0) {
				if (term.cvalue < lastValue) {
					term.rank++;
					rank++;
				}
			}
			lastValue = term.cvalue;
		}
		preferences.sortedColumn = sc;
		preferences.sortPrefs[TermoPLConsts.CVALUE] = s;
		
		if (termsView != null) {
			String str = termsView.getSearchString();
			
			termsView.setSearchString("");
			termsView.search(str);
		}
		trim(preferences.trimResults, preferences.multiWordTermsOnly, preferences.filterResults);
	}
	
	public void showResults()
	{
		if (termsView != null) {
			termsView.showProgress(false);
			if (cancelled) {
				termsView.setWarning(TermsView.CANCEL_WARNING);
				termsView.setAccessory(extractAction);
			}
			else if (nterms > 0) {
				termsView.setInfo(getStat());
				termsView.resetResults();
			}
			else {
				termsView.setWarning(TermsView.SEARCH_FAILED);
				termsView.setAccessory(null);
			}
		}
	}
	
	public void setCTerms(boolean calcSimplifiedForms)
	{
		if (cterms != null)	{
			for (Term t : terms) {
				String str;
				
				if (calcSimplifiedForms) str = calcSimplifiedForm(t);
				else str = t.str;
				t.setCTerm(cterms.get(str));
			}
		}
	}
	
	public void applyContrastiveRanking()
	{
		if (cterms != null) {
			double s = 0.0, S = 0.0;
			boolean useCValues = (preferences.contrastiveRankingMethod == 4 ? false : preferences.useCValues);
			
			for (Term t : terms) {
				t.setStatus(Term.DEFAULT_STATUS);
				s += (useCValues ? t.cvalue : (double)t.freq_s);
			}
			for (Term t : cterms.values()) {
				S += (useCValues ? t.cvalue : (double)t.freq_s);
			}
			switch (preferences.contrastiveRankingMethod) {
				case 1 : computeLL(s, S, !preferences.useCValues); break;
				case 2 : computeTFITF(s, S, !preferences.useCValues); break;
				case 3 : computeCSmw(s, S, !preferences.useCValues); break;
				case 4 : computeTW(s, S);
			}
		}
	}
	
	public void computeLL(double s, double S, boolean useFrq)
	{
		for (Term t : terms) {
			double F;
			Term ct = t.getCTerm();
			
			if (ct == null) F = 0.0;
			else {
				if (useFrq) F = (double)ct.freq_s;
				else F = ct.cvalue;
			}
			t.contrast = F;
			if ((preferences.applyContrastiveRankingForTopRankedTerms && t.cvalue <= preferences.mincvalue) || 
				(preferences.applyContrastiveRankingForFrequentTerms && t.freq_s <= preferences.minfrq))
			{
				t.setStatus(Term.IGNORE_STATUS);
			}
		}
		
		for (Term t : terms) {
			if (t.getStatus() != Term.IGNORE_STATUS) {
				double Ef, EF, r1, r2;
				
				r1 = (useFrq ? (double)t.freq_s : t.cvalue);
				r2 = t.contrast;
				
				Ef = s * (r1 + r2) / (s + S);
				EF = S * (r1 + r2) / (s + S);
				if (r1 == 0.0 && r2 == 0.0) t.contrast = 0.0;
				else if (r1 == 0.0 || r2 == 0.0) t.contrast = Double.NEGATIVE_INFINITY;
				else t.contrast = 2 * (r1 * Math.log10(r1 / Ef) + r2 * Math.log10(r2 / EF));
				r1 = 100000.0 * r1 / s;
				r2 = 100000.0 * r2 / S;
				if (r2 == 0) t.setStatus(Term.UNIQUE_STATUS);
				else t.setStatus(getStatus(r1, r2));
			}
			else t.contrast = 0.0;
		}
	}
	
	public void computeTFITF(double s, double S, boolean useFrq)
	{
		double N = s;
		
		for (Term t : terms) {
			double F;
			Term ct = t.getCTerm();
			
			if (ct == null) F = 0.0;
			else {
				if (useFrq) F = (double)ct.freq_s;
				else F = ct.cvalue;
			}
			t.contrast = F;
			N += F;
			if ((preferences.applyContrastiveRankingForTopRankedTerms && t.cvalue <= preferences.mincvalue) || 
					(preferences.applyContrastiveRankingForFrequentTerms && t.freq_s <= preferences.minfrq))
			{
				t.setStatus(Term.IGNORE_STATUS);
			}
		}
		
		for (Term t : terms) {
			if (t.getStatus() != Term.IGNORE_STATUS) {
				double r1, r2;
				
				r1 = (useFrq ? (double)t.freq_s : t.cvalue);
				r2 = t.contrast;
				if (r1 == 0.0) t.contrast = Double.NEGATIVE_INFINITY;
				else t.contrast = Math.log10((double)r1) * Math.log10(N / (r1 + r2));
				r1 = 100000.0 * r1 / s;
				r2 = 100000.0 * r2 / S;
				if (r2 == 0) t.setStatus(Term.UNIQUE_STATUS);
				else t.setStatus(getStatus(r1, r2));
			}
			else t.contrast = 0.0;
		}
	}
	
	public void computeCSmw(double s, double S, boolean useFrq)
	{
		double N = 0.0;
		
		for (Term t : terms) {
			double F;
			Term ct = t.getCTerm();
			
			if (ct == null) F = 0.0;
			else {
				if (useFrq) F = (double)ct.freq_s;
				else F = ct.cvalue;
			}
			t.contrast = F;
			N += F;
			if ((preferences.applyContrastiveRankingForTopRankedTerms && t.cvalue <= preferences.mincvalue) || 
					(preferences.applyContrastiveRankingForFrequentTerms && t.freq_s <= preferences.minfrq))
				{
					t.setStatus(Term.IGNORE_STATUS);
				}
		}
		
		for (Term t : terms) {
			if (t.getStatus() != Term.IGNORE_STATUS) {
				double r1, r2;
				
				r1 = (useFrq ? (double)t.freq_s : t.cvalue);
				r2 = t.contrast;
				if (r1 == 0.0) t.contrast = Double.NEGATIVE_INFINITY;
				else if (r2 == 0.0) t.contrast = Double.POSITIVE_INFINITY;
				else t.contrast = Math.log10(Math.log10(r1) * N * r1 / r2);
				r1 = 100000.0 * r1 / s;
				r2 = 100000.0 * r2 / S;
				if (r2 == 0) t.setStatus(Term.UNIQUE_STATUS);
				else t.setStatus(getStatus(r1, r2));
			}
			else t.contrast = 0.0;
		}
	}
	
	public void computeTW(double s, double S)
	{
		for (Term t : terms) {
			double F;
			Term ct = t.getCTerm();
			
			if (ct == null) F = 0.0;
			else F = (double)ct.freq_s;
			t.contrast = F;
			if ((preferences.applyContrastiveRankingForTopRankedTerms && t.cvalue <= preferences.mincvalue) || 
				(preferences.applyContrastiveRankingForFrequentTerms && t.freq_s <= preferences.minfrq))
			{
				t.setStatus(Term.IGNORE_STATUS);
			}
			else {
				double r1 = t.freq_s, r2 = F;
				
				r1 = 100000.0 * r1 / s;
				r2 = 100000.0 * r2 / S;
				if (r2 == 0) t.setStatus(Term.UNIQUE_STATUS);
				else t.setStatus(getStatus(r1, r2));
			}
		}
		
		double[] DCArray = new double[terms.length];
		double maxDC = 0.0;
		int i = 0;
		
		for (Term t : terms) {
			double freqt = 0.0, DC = 0.0;
			
			for (DocFreq dc : t.getDocFreq()) {
				freqt += dc.freq;
			}
			
			for (DocFreq dc : t.getDocFreq()) {
				double pt = dc.freq / freqt;
				
				DC += pt * Math.log10(pt);
			}
			DC = -DC;
			DCArray[i++] = DC;
			if (maxDC < DC) maxDC = DC;
		}
		
		i = 0;
		if (maxDC == 0.0) maxDC = 1.0;
		for (Term t : terms) {
			double DR, p, P;
			
			p = t.freq_s / s;
			P = t.contrast / S;
			DR = p / Math.max(p, P);
			t.contrast = preferences.alpha * DR + preferences.beta * DCArray[i++] / maxDC;
		}
	}
	
	public byte getStatus(double f, double F)
	{
		double result;
		
		if (f >= F) result = f / F;
		else result = -F / f;
		if (result < -9) return 0;
		if (result >= -9 && result < -7) return 1;
		if (result >= -7 && result < -5) return 2;
		if (result >= -5 && result < -3) return 3;
		if (result >= -3 && result < -1) return 4;
		if (result >= 1 && result < 3) return 5;
		if (result >= 3 && result < 5) return 6;
		if (result >= 5 && result < 7) return 7;
		if (result >= 7 && result < 9) return 8;
		return 9;
	}
	
	public String calcBaseForm(Term t, BaseFormGuesser guesser)
	{
		return guesser.calcBaseForm(t);
	}
	
	public void calculateBaseForms()
	{
		Thread th;
		
		if (extractor == null) {
			if (termsView != null) {
				termsView.showProgress(true);
			}
		}
		changeProgress(5);
		termQueue = new ArrayBlockingQueue<Term>(MAX_TERM_QUEUE_SIZE);
		countDown = new CountDownLatch(MAX_THREADS + 1);
		th = new ProcessTerms();
		th.start();
		
        Thread bft[] = new Thread[MAX_THREADS];
        
		for (int i = 0; i < MAX_THREADS; i++) {
			bft[i] = new BaseFormThread();
		}
		for (int i = 0; i < MAX_THREADS; i++) {
			bft[i].start();
		}
		try {
			countDown.await();
		} 
		catch (InterruptedException e) {
			e.printStackTrace();
		}
		if (extractor == null) {
			if (termsView != null) {
				termsView.showProgress(false);
			}
		}
		if (termsView != null) termsView.search();
	}
	
	public String calcSimplifiedForm(Term t)
	{
		Form f = t.getForms().getFirst();
		StringBuffer buffer = new StringBuffer();
		Token prev = null;
		int index = 1, prevIndex = -1;
		
		for (MatchedToken mt : f.getTokens()) {
			Token tok = mt.token;
			
			if (prev != null) {
				if (tok instanceof UDToken) index = ((UDToken) tok).index;
				else if (tok instanceof MultiWordToken) index = ((MultiWordToken) tok).endToken();
				else index = prevIndex + 1;
				if (prev.spaceAfter || index > prevIndex + 1) {
					buffer.append(" ");
					buffer.append(tok.lemma);
				}
				else buffer.append(tok.lemma);
			}
			else buffer.append(tok.lemma);
			prev = tok;
			prevIndex = index;
		}
		return buffer.toString();
	}
	
	public void calculateSimplifiedForms()
	{
		Thread th;
		
		changeProgress(5);
		termQueue = new ArrayBlockingQueue<Term>(MAX_TERM_QUEUE_SIZE);
		countDown = new CountDownLatch(MAX_THREADS + 1);
		th = new ProcessTerms();
		th.start();
		for (int i = 0; i < MAX_THREADS; i++) {
			th = new SimplifiedFormThread();
			th.start();
		}
		try {
			countDown.await();
		} 
		catch (InterruptedException e) {
			e.printStackTrace();
		}
		if (termsView != null) termsView.search();
	}
	
	public void recalculateForms()
	{
		if (termsView != null) {
			termsView.showProgress(true, false);
		}
		if (preferences.calculateBaseForms) calculateBaseForms();
		else calculateSimplifiedForms();
		if (preferences.sortedColumn == TermoPLConsts.TERM) Arrays.sort(terms, 0, tableSize, sorter);
		if (termsView != null) {
			termsView.showProgress(false);
		}
	}
	
	public void sortTable()
	{
		Arrays.sort(terms, sorter);		
	}
	
	public void sortTable(int first, int num)
	{
		Arrays.sort(terms, first, num, sorter);	
	}
	
	public void sortTable(int column)
	{
		if (preferences.sortedColumn == column) 
			preferences.sortPrefs[column] = -preferences.sortPrefs[column];
		else preferences.sortedColumn = column;
		TermoPL.preferences.sortedColumn = preferences.sortedColumn;
		TermoPL.preferences.setModified(true);
		sortTable(0, tableSize);
	}
	
	public void changeProgress(int type)
	{
		if (!TermoPL.batchMode && termsView != null) termsView.changeProgress(type);
	}
	
	public void reportTagging(String fName)
	{
		if (!TermoPL.batchMode && termsView != null) termsView.reportTagging(fName);
	}
	
	public void reportPreprocessing(String fName)
	{
		if (!TermoPL.batchMode && termsView != null) termsView.reportPreprocessing(fName);
	}
	
	public void report(String fName, int count)
	{
		if (!TermoPL.batchMode && termsView != null) termsView.report(fName, count);
	}
	
	public void report(int count, float progress)
	{
		if (!TermoPL.batchMode && termsView != null) termsView.report(count, progress);
	}
	
	public void report(int processed, int max, float value)
	{
		if (!TermoPL.batchMode && termsView != null) termsView.report(processed, max, value);
	}
	
	public void report(float value)
	{
		if (!TermoPL.batchMode && termsView != null) termsView.report(value);
	}
	
	public void setAcceptDET(int f)
	{
		acceptDET = f;
	}
	
	public int acceptDET()
	{
		if (preferences.useUD) return acceptDET;
		return -1;
	}
	
	public boolean isLoaded()
	{
		return loaded;
	}
	
	public boolean isCancelled()
	{
		return cancelled;
	}
	
	public boolean isCorrupted()
	{
		return corrupted;
	}
	
	public Preferences getPreferences()
	{
		return preferences;
	}

	public File[] getSelectedFiles()
	{
		return selectedFiles;
	}
	
	public File[] getOldFiles()
	{
		return oldFiles;
	}
	
	public File[] getNewFiles()
	{
		return newFiles;
	}
	
	public LinkedList<FileDescr> getAnalyzedFiles()
	{
		return analyzedFiles;
	}
	
	public FileDescr getAnalyzedFile(int fileID)
	{
		FileDescr descr = analyzedFiles.get(fileID);
		
		if (descr.type == CorpusReader.UNKNOWN_FORMAT) {
			descr = new FileDescr(new File(descr.file.getPath() + ".trm"), CorpusReader.CONLLU_FORMAT);
		}
		return descr;
	}
	
	public Set<String> getStopWords()
	{
		if (preferences.checkStopWords)	{
			if (preferences.stopWords != null) {
				if (preferences.stopWordSet == null) {
					preferences.stopWordSet = new HashSet<String>();
					preferences.stopWordSet.addAll(preferences.stopWords);
				}
				return preferences.stopWordSet;
			}
		}
		return null;
	}
	
	public Set<String> getCommonTerms()
	{
		if (preferences.removeCommonTerms) {
			if (preferences.commonTerms != null) {
				if (preferences.commonTermSet == null) {
					preferences.commonTermSet = new HashSet<String>();
					for (CommonTerm ct : preferences.commonTerms) {
						preferences.commonTermSet.add(ct.sf);
					}
				}
				return preferences.commonTermSet;
			}
		}
		return null;
	}
	
	public Template getCompoundPrepositions()
	{
		if (preferences.removeCompoundPreps) return preferences.compoundPrepositions;
		return null;
	}
	
	public Template getSearchTemplate()
	{
		return preferences.template;
	}
	
	public boolean useDocID()
	{
		return (preferences.applyContrastiveRanking && preferences.contrastiveRankingMethod == 4);
	}
	
	public int changeDocID(int docNumber)
	{
		if (useDocID()) return docNumber + 1;
		return docNumber;
	}
	
	public Term[] getTerms()
	{
		return terms;
	}
	
	public HashMap<String, Term> getTermMap()
	{
		return termMap;
	}
	
	public HashMap<String, Term> getCTerms()
	{
		return cterms;
	}
	
	public int getTableSize()
	{
		return tableSize;
	}
	
	public TermsView getTermsView()
	{
		return termsView;
	}
	
	public FormsView getFormsView()
	{
		return formsView;
	}
	
	public SentencesView getSentencesView()
	{
		return sentencesView;
	}
	
	public GroupsView getGroupsView()
	{
		return groupsView;
	}
	
	public void changeFontSize()
	{
		if (termsView != null) termsView.changeFontSize();
		if (formsView != null) formsView.changeFontSize();
		if (sentencesView != null) sentencesView.changeFontSize();
		if (groupsView != null) groupsView.changeFontSize();
	}
	
	private class ColumnSorter implements Comparator<Term>
	{
		
		private Collator collator = Collator.getInstance(new Locale("pl", "PL"));
		
		public int compare(Term t1, Term t2) 
		{
			switch (preferences.sortedColumn) {
				case TermoPLConsts.RANK :
					return preferences.sortPrefs[preferences.sortedColumn] * (t1.rank - t2.rank);
				case TermoPLConsts.TERM : 
					return preferences.sortPrefs[preferences.sortedColumn] * collator.compare(t1.str, t2.str);
				case TermoPLConsts.CVALUE : 
					return preferences.sortPrefs[preferences.sortedColumn] * (int)Math.signum(t1.cvalue - t2.cvalue);
				case TermoPLConsts.CSVALUE : 
					return preferences.sortPrefs[preferences.sortedColumn] * (int)Math.signum(t1.contrast - t2.contrast);
				case TermoPLConsts.LENGTH : 
					return preferences.sortPrefs[preferences.sortedColumn] * (t1.len - t2.len);
				case TermoPLConsts.FREQS : 
					return preferences.sortPrefs[preferences.sortedColumn] * (t1.freq_s - t2.freq_s);
				case TermoPLConsts.FREQIN : 
					return preferences.sortPrefs[preferences.sortedColumn] * (t1.freq_in - t2.freq_in);
				case TermoPLConsts.CONTEXT : 
					return preferences.sortPrefs[preferences.sortedColumn] * (t1.lk - t2.lk);
			}
			return 0;
		}
		
	}

	private class LoadThread extends Thread
	{
		
		private File input;
		
		public LoadThread(File input)
		{
			this.input = input;
		}
		
		public void run()
		{
			load(input);
		}
		
		@SuppressWarnings("unchecked")
		public void load(File input)
		{
			ObjectInputStream is;
			int ncterms;
			
			try {
				cancelled = false;
				is = new ObjectInputStream(new FileInputStream(input));
				formsCollected = is.readBoolean();
				sentencesIndexed = is.readBoolean();
				nsentences = is.readInt();
				ntokens = is.readInt();
				nterms = is.readInt();
				ncterms = is.readInt();
				rebuild = is.readInt();
				corpusName = (String)is.readObject();
				preferences = (Preferences)is.readObject();
				preferences.workSpace = TermoPL.preferences.workSpace;
				preferences.finalizeCreate(true);
				
				String[] ftable = (String[])is.readObject();
				
				if (ftable == null) selectedFiles = null;
				else {
					selectedFiles = new File[ftable.length];
					
					for (int i = 0; i < ftable.length; i++)
						selectedFiles[i] = new File(preferences.resolvePath(ftable[i]));
				}
				ftable = (String[])is.readObject();
				if (ftable == null) newFiles = null;
				else {
					newFiles = new File[ftable.length];
					
					for (int i = 0; i < ftable.length; i++)
						newFiles[i] = new File(preferences.resolvePath(ftable[i]));
				}
				ftable = (String[])is.readObject();
				if (ftable == null) oldFiles = null;
				else {
					oldFiles = new File[ftable.length];
					
					for (int i = 0; i < ftable.length; i++)
						oldFiles[i] = new File(preferences.resolvePath(ftable[i]));
				}
				
				ArrayList<Pair<String, Integer>> aFiles = (ArrayList<Pair<String, Integer>>)is.readObject();
				
				if (aFiles == null) analyzedFiles = null;
				else {
					analyzedFiles = new LinkedList<FileDescr>();
					for (Pair<String, Integer> pair : aFiles)
						analyzedFiles.add(new FileDescr(new File(preferences.resolvePath(pair.first)), pair.second));
				}

				if (!cancelled) {
					if (nterms == 0) terms = null;
					else {
						terms = new Term[nterms];
						for (int i = 0; i < nterms && !cancelled; i++) terms[i] = (Term)is.readObject();
					}
				}
				if (!cancelled) {
					if (ncterms == 0) cterms = null;
					else {
						cterms = new HashMap<String, Term>(ncterms);
						for (int i = 0; i < ncterms && !cancelled; i++) {
							String str = (String)is.readObject();
							Term t = (Term)is.readObject();
							
							cterms.put(str, t);
						}
					}
				}
				if (!cancelled) {
					preferences.finalizeCreate(true);
					if (termsView != null) termsView.finishCreate();
				}
				is.close();
				if (cancelled) remove();
				else {
					trim(preferences.trimResults, preferences.multiWordTermsOnly, preferences.filterResults);
					if (termsView != null) {
						termsView.resetResults();
						termsView.showProgress(false);
						switch (rebuild) {
							case 0 :
								termsView.setInfo(getStat());
								break;
							case 1 :
							case 2 :
								termsView.setWarning(TermsView.EXTRACT_WARNING);
								termsView.setAccessory(extractAction);
								break;
							case 3 :
								termsView.setWarning(TermsView.SELECT_WARNING);
								termsView.setAccessory(selectAction);
						}
					}
					loaded = true;
				}
			}
			catch (Exception exception) {
				corrupted = true;
				remove();
				if (TermoPL.batchMode) System.out.println("File has wrong format or is corrupted -- " + input.getPath());
				else JOptionPane.showMessageDialog(TermoPL.dialogOwner, "File has wrong format or is corrupted.", "Error", JOptionPane.ERROR_MESSAGE);
			}
			finishLoading();
			TermoPL.application.finishOpening(TermoPLDocument.this);
		}
		
	}
	
	private class ContrastiveDataLoader extends Thread
	{
		
		private File input;
		
		public ContrastiveDataLoader(String path)
		{
			input = new File(path);
		}
		
		public void run()
		{
			HashMap<String, Term> newCTerms = null;
			
			changeProgress(0);
			newCTerms = loadContrastiveTerms(input);
			if (newCTerms != null) cterms = newCTerms;
		}
		
	}
	
	private class MergeThread extends Thread
	{

		private File file;
		private boolean discardGroups;
		
		public MergeThread(File file)
		{
			this.file = file;
			termMap = new HashMap<String, Term>();
			discardGroups = false;
		}
		
		public void run()
		{
			boolean bf = preferences.calculateBaseForms;
			
			corrupted = false;
			cancelled = false;
			for (Term t : terms) {
				if (t instanceof TermEx) discardGroups = true;
				if (cancelled) break;
				
				String key = (bf ? calcSimplifiedForm(t) : t.str);
				
				termMap.put(key, t);
			}
			if (!cancelled) merge();
		}
		
		@SuppressWarnings("unchecked")
		public void merge()
		{
			ObjectInputStream is;
			Preferences prefs;
			String[] files;
			LinkedList<File> fileList;
			ArrayList<Pair<String, Integer>> af;
			BaseFormGuesser guesser = null;
			Term term;
			int snt = 0, tok = 0, trm = 0;
			int count, maxcount;
			boolean bf;
			
			try {
				is = new ObjectInputStream(new FileInputStream(file));
				is.readBoolean();
				is.readBoolean();
				snt = is.readInt();
				tok = is.readInt();
				trm = is.readInt();
				is.readInt();
				is.readInt();
				is.readObject();
				prefs = (Preferences)is.readObject();
				files = (String[])is.readObject();
				fileList = new LinkedList<File>();
				if (selectedFiles != null) {
					for (File sf : selectedFiles) fileList.add(sf);
				}
				if (files != null) {
					for (String f : files) {
						String rf = prefs.resolvePath(f);
						
						if (selectedFiles != null) {
							for (File sf : selectedFiles) {
								if (!rf.equals(sf.getAbsolutePath())) fileList.add(new File(rf));
							}
						}
						else fileList.add(new File(rf));
					}
				}
				if (fileList.isEmpty()) selectedFiles = null;
				else selectedFiles = fileList.toArray(new File[0]);
				files = (String[])is.readObject();
				fileList = new LinkedList<File>();
				if (newFiles != null) {
					for (File nf : newFiles) fileList.add(nf);
				}
				if (files != null) {
					for (String f : files) {
						String rf = prefs.resolvePath(f);
						
						if (newFiles != null) {
							for (File nf : newFiles) {
								if (!rf.equals(nf.getAbsolutePath())) fileList.add(new File(rf));
							}
						}
						else fileList.add(new File(rf));
					}
				}
				if (fileList.isEmpty()) newFiles = null;
				else newFiles = fileList.toArray(new File[0]);
				files = (String[])is.readObject();
				fileList = new LinkedList<File>();
				if (oldFiles != null) {
					for (File of : oldFiles) fileList.add(of);
				}
				if (files != null) {
					for (String f : files) {
						String rf = prefs.resolvePath(f);
						
						if (oldFiles != null) {
							for (File of : oldFiles) {
								if (!rf.equals(of.getAbsolutePath())) fileList.add(new File(rf));
							}
						}
						else fileList.add(new File(rf));
					}
				}
				if (fileList.isEmpty()) oldFiles = null;
				else oldFiles = fileList.toArray(new File[0]);
				af = (ArrayList<Pair<String, Integer>>) is.readObject();
				bf = prefs.calculateBaseForms;
				allowCollectingForms(preferences.collectAllForms && prefs.collectAllForms);
				allowIndexingSentences(preferences.makeIndex && prefs.makeIndex);
				maxcount = trm / 10;
				count = 0;
				for (int i = 0; i < trm && !cancelled; i++) {
					Term t = (Term)is.readObject();
					String key = (bf ? calcSimplifiedForm(t) : t.str);
					
					if (t instanceof TermEx) discardGroups = true;
					if (preferences.calculateBaseForms && !bf) {
						if (guesser == null) {
							if (preferences.useUD) guesser = new BaseFormGuesser();
							else guesser = new BaseFormGuesser(preferences.tagset);
						}
						t.str = calcBaseForm(t, guesser);
					}
					else if (!preferences.calculateBaseForms && bf)
						t.str = key;
					
					term = termMap.get(key);
					if (term == null) termMap.put(key, t);
					else {
						term.addContexts(t.getContexts());
						if (preferences.collectAllForms) term.addForms(t.getForms(), preferences.collectAllForms);
						else term.registerForms(t);
						if (preferences.makeIndex) term.addSentenceRef(t.getSentenceRef(), analyzedFiles.size());
						term.addDocFreq(t.getDocFreq());
					}
					count++;
					if (count >= maxcount) {
						count = 0;
						if (termsView != null) {
							termsView.report((float)i / (float)trm);
						}
					}
				}
				if (termsView != null) termsView.report(1.0F);
				is.close();
				if (!cancelled) {
					terms = termMap.values().toArray(new Term[0]);
					
					if (discardGroups) {
						for (int i = 0; i < terms.length; i++) {
							Term t = terms[i];
							
							if (t instanceof TermEx) terms[i] = t.copy();
						}
					}
					nterms = terms.length;
					tableSize = nterms;
					nsentences += snt;
					ntokens += tok;
					calculate(preferences.cntxMethod);
					
					if (preferences.applyContrastiveRanking && cterms != null) {
						setCTerms(false);
						applyContrastiveRanking();	
					}
					prepareTable();
					for (Pair<String, Integer> pair : af)
						analyzedFiles.add(new FileDescr(new File(preferences.resolvePath(pair.first)), pair.second));
					preferences.makeGroups = false;
					modified = true;
				}
			}
			catch (Exception exception) {
				corrupted = true;
				if (TermoPL.batchMode) System.out.println("File has wrong format or is corrupted -- " + file.getPath());
				else JOptionPane.showMessageDialog(TermoPL.dialogOwner, "File has wrong format or is corrupted.", "Error", JOptionPane.ERROR_MESSAGE);
			}
			finalizeMerge();
		}
		
	}
	
	private class ExtractorThread extends Thread
	{
		
		private ExtractorEngine extractor;
		private GroupingEngine groupingEngine;
		private Finalizer finalizer;
		private ContrastiveDataLoader loader;
		
		public ExtractorThread(ExtractorEngine extractor, GroupingEngine groupingEngine, 
				Finalizer finalizer, ContrastiveDataLoader loader)
		{
			this.extractor = extractor;
			this.groupingEngine = groupingEngine;
			this.finalizer = finalizer;
			this.loader = loader;
		}
		
		public void run()
		{
			searching = true;
			if (loader != null) {
				loader.start();
				try {
					loader.join();
				}
				catch (InterruptedException exception) {
				}
			}
			if (!cancelled) extractor.start();
			try {
				extractor.join();
			}
			catch (InterruptedException exception) {
			}
			if (!cancelled && !interrupted) {
				termMap = extractor.getTermMap();
				terms = extractor.getTerms();
				nterms = extractor.getNumberOfTerms();
				if (groupingEngine != null) {
					groupingEngine.start();
					try {
						groupingEngine.join();
					}
					catch (InterruptedException exception) {
					}
				}
			}
			finalizer.start();
			if (TermoPL.batchMode) {
				try {
					finalizer.join();
				}
				catch (InterruptedException exception) {
				}
			}
		}
		
	}
	
	private class GroupingEngine extends Thread
	{
		
		public void run()
		{
			makeGroups();
		}
		
	}
	
	private class Finalizer extends Thread
	{
		
		public void run()
		{
			if (!cancelled && !interrupted) {
				LinkedList<FileDescr> aFiles;
				
				nsentences += extractor.getNumberOfSentences();
				ntokens += extractor.getNumberOfTokens();
				aFiles = extractor.getAnalyzedFiles();
				formsCollected = preferences.collectAllForms;
				sentencesIndexed = preferences.makeIndex;
				if (terms != null) {
					tableSize = nterms;
					calculate(preferences.cntxMethod);
					if (preferences.applyContrastiveRanking && cterms != null) {
						setCTerms(false);
						applyContrastiveRanking();	
					}
					if (preferences.calculateBaseForms) calculateBaseForms();
					if (!cancelled) prepareTable();
					else {
						termMap = null;
						terms = null;
						tableSize = 0;
					}
				}
				else tableSize = 0;
				if (rebuild == 1) {
					analyzedFiles.addAll(aFiles);
				}
				else { // rebuild == 2
					analyzedFiles = aFiles;
				}
				selectedFiles = new File[analyzedFiles.size()];
				
				int i = 0;
				
				for (FileDescr fd : analyzedFiles) selectedFiles[i++] = fd.file;
				oldFiles = selectedFiles;
				newFiles = null;
				rebuild = 0;
			}
			else {
				terms = null;
				tableSize = 0;
				termMap = null;
			}
			if (terms != null) {
				if (!(terms[0] instanceof TermEx)) termMap = null;
			}
			extractor = null;
			preferences.extract = false;
			preferences.compare = false;
			preferences.reloadContrastiveTerms = false;
			preferences.repaint = false;
			showResults();
			searching = false;
		}
	}
	
	private class CompareThread extends Thread
	{
		
		private ContrastiveDataLoader loader;
		
		public CompareThread(ContrastiveDataLoader loader)
		{
			this.loader = loader;
		}
		
		public void run()
		{
			if (loader != null) {
				loader.start();
				try {
					loader.join();
				}
				catch (InterruptedException exception) {}
			}
			if (cterms != null) {
				setCTerms(preferences.calculateBaseForms);
				applyContrastiveRanking();
			}
			finishCompare();
		}
		
	}
	
	private class SaveThread extends Thread
	{
		
		private File file;
		
		public SaveThread(File file)
		{
			this.file = file;
		}
		
		public void run()
		{
			try {
				ObjectOutputStream output = new ObjectOutputStream(new FileOutputStream(file));
				
				save(output);
			}
			catch (IOException exception) {
				if (TermoPL.batchMode) System.err.println("Error occured while writing a file.");
				else JOptionPane.showMessageDialog(TermoPL.dialogOwner, "Error occured while writing a file.", "Error", JOptionPane.ERROR_MESSAGE);
			}
			if (termsView != null) termsView.showProgress(false);
		}
		
	}
	
	private class ExportThread extends Thread
	{
		
		private PrintWriter pw;
		private int exportType;
		
		public ExportThread(PrintWriter pw, int exportType)
		{
			this.pw = pw;
			this.exportType = exportType;
		}
		
		public void run()
		{
			export(pw, exportType);
			if (termsView != null) termsView.showProgress(false);
		}
		
	}
	
	private class ProcessTerms extends Thread
	{
		
		public ProcessTerms()
		{
			super();
		}
		
		public void run()
		{
			int count = 0, count1 = 0, count2 = 0, len = terms.length;
			
			for (Term term : terms) {
				if (cancelled) break;
				
				try {
					termQueue.put(term);
				}
				catch (InterruptedException exception) {
				}
				count++;
				if (++count1 >= 1000) {
					count1 = 0;
					if (++count2 >= 25000) {
						count2 = 0;
						System.gc();
					}
					report(count, len, (float)count / (float)len);
				}
			}
			report(count, len, 1.0F);
			try {
				termQueue.put(DUMMY_TERM);
			}
			catch (InterruptedException exception) {
			}
			countDown.countDown();
		}
		
	}
	
	private class BaseFormThread extends Thread
	{
		
		private BaseFormGuesser baseFormGuesser;
		
		public BaseFormThread()
		{
			super();
			if (preferences.language.equals("pl") && !preferences.useCustomTagset)
				baseFormGuesser = new BaseFormGuesser(preferences.tagset);
			else baseFormGuesser = new BaseFormGuesser();
		}
		
		public void run()
		{
			boolean done = false;
			
			try {
				while (!done) {
					Term term = termQueue.take();
					
					if (term == DUMMY_TERM) {
						termQueue.put(term);
						done = true;
					}
					else {
						term.str = calcBaseForm(term, baseFormGuesser);
					}
				}
			}
			catch (InterruptedException exception) {
			}
			countDown.countDown();
		}
		
	}
	
	private class SimplifiedFormThread extends Thread
	{
		
		public void run()
		{
			boolean done = false;
			
			try {
				while (!done) {
					Term term = termQueue.take();
					
					if (term == DUMMY_TERM) {
						termQueue.put(term);
						done = true;
					}
					else {
						term.str = calcSimplifiedForm(term);
					}
				}
			}
			catch (InterruptedException exception) {
			}
			countDown.countDown();
		}
		
	}
	
	private class FileNameFilter implements FilenameFilter
	{
		
		private String namePattern;
		
		public FileNameFilter(String namePattern)
		{
			namePattern = namePattern.replace(".", "\\.");
			namePattern = namePattern.replace("?", ".");
			namePattern = namePattern.replace("*", ".*");
			this.namePattern = namePattern;
		}

		public boolean accept(File dir, String name) 
		{
			return name.matches(namePattern);
		}
		
	}
	
}
