package pl.waw.ipipan.zil.core.textSelector;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Scanner;
import java.util.Timer;
import java.util.TimerTask;

import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;

public class TextSelector extends JFrame implements Runnable {

	private static final String CURRENT_INDEX = "currentIndex.txt";
	private static final String TEXT_SELECTOR = "Text selector";
	private static final int WIDTH = 800;
	private static final int HEIGTH = 600;
	private static final long serialVersionUID = 276545319462523786L;
	private static final Logger logger = Logger.getLogger(TextSelector.class);

	private List<File> files;
	private File currentIndexFile;
	private File dirForRejects;

	private int currentIndex;
	private JTextArea textArea;

	public TextSelector(File dirWithTexts, File dirForRejects) {
		this.currentIndexFile = new File(dirWithTexts + File.separator + CURRENT_INDEX);
		this.dirForRejects = dirForRejects;

		this.files = recFindTexts(dirWithTexts);
		logger.info("Found " + files.size() + " texts.");

		if (files.size() == 0) {
			logger.error("No texts to load.");
			return;
		}
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) {

		if (args.length != 2) {
			logger.error("Wrong usage! Try: java -jar " + TextSelector.class.getSimpleName()
					+ " dirWithText dirForRejects");
			return;
		}

		File dirWithTexts = new File(args[0]);
		File dirForRejects = new File(args[1]);

		if (!dirWithTexts.exists() || !dirForRejects.exists()) {
			logger.error("One of directories doesn't exist!");
			return;
		}

		final TextSelector selector = new TextSelector(dirWithTexts, dirForRejects);
		try {
			SwingUtilities.invokeAndWait(selector);
		} catch (Exception e) {
			logger.error("Error starting application: " + e.getLocalizedMessage());
			e.printStackTrace();
			return;
		}
	}

	private static List<File> recFindTexts(File dirWithTexts) {
		List<File> result = new ArrayList<File>();

		List<File> files = new ArrayList<File>(Arrays.asList(dirWithTexts.listFiles()));
		Collections.sort(files);
		for (File f : files) {
			if (f.isDirectory())
				result.addAll(recFindTexts(f));
			else if (f.getName().matches("text.*\\.xml")) {
				result.add(f);
				break;
			}
		}
		return result;
	}

	public void run() {
		this.setTitle(TEXT_SELECTOR);
		this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		this.setSize(WIDTH, HEIGTH);
		Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		this.setLocation((screenSize.width - WIDTH) / 2, (screenSize.height - HEIGTH) / 2);
		this.setVisible(true);

		textArea = new JTextArea();
		textArea.setLineWrap(true);
		textArea.setWrapStyleWord(true);
		this.getContentPane().add(new JScrollPane(textArea));

		MainWindowListener l = new MainWindowListener();
		this.addWindowListener(l);
		textArea.addKeyListener(l);

		int index = loadCurrentTextIndex();
		loadText(index);
	}

	private int loadCurrentTextIndex() {
		int index = 0;
		try {
			index = new Scanner(currentIndexFile).nextInt();
			logger.info("Loaded index " + index);
		} catch (Exception ex) {
			logger.info("Using default index.");
		}
		return index;
	}

	private void loadText(int i) {		
		File f = files.get(i);
		currentIndex = i;
		this.setTitle(TEXT_SELECTOR + " " + f + " (" + currentIndex + "/" + files.size() + ")");

		textArea.setText(XmlIO.loadText(f));
	}

	private void loadNext() {
		if (files.size() > currentIndex + 1)
			loadText(currentIndex + 1);
		else {
			loadText(currentIndex);
			logger.info("Last text in dir.");
		}
	}

	private void loadPrev() {
		if (currentIndex > 0)
			loadText(currentIndex - 1);
		else {
			loadText(currentIndex);
			logger.info("First text in dir.");
		}
	}

	private void requestExit() {
		logger.info("Exiting...");

		try {
			FileWriter w = new FileWriter(currentIndexFile);
			w.write(Integer.toString(currentIndex));
			w.close();
			logger.info("Saved index in file: " + currentIndexFile);
		} catch (IOException e) {
			logger.error("Error saving index.");
		}

		System.exit(0);
	}

	private void rejectText() {		
		changeTextAreaColor(Color.ORANGE);
		File file = files.get(currentIndex);
		File parentFile = file.getParentFile();
		logger.info("Rejecting text: " + file);
		try {
			FileUtils.moveDirectory(parentFile, new File(dirForRejects + File.separator + parentFile.getName()));
		} catch (IOException e) {
			logger.error("Error rejecting file " + file);
			return;
		}

		files.remove(currentIndex);
		currentIndex--;
		loadNext();
	}

	private void saveText() {		
		changeTextAreaColor(Color.GREEN);
		
		String text = textArea.getText();
		String[] spl = text.split("\n\n");

		XmlIO.saveText(files.get(currentIndex), spl);
	}

	private void changeTextAreaColor(Color color) {
		textArea.setBackground(color);
		Timer t = new Timer();
		t.schedule(new TimerTask() {			
			@Override
			public void run() {
				textArea.setBackground(Color.WHITE);				
			}
		}, 1000*1);
		
	}

	private class MainWindowListener extends WindowAdapter implements KeyListener {

		public void windowClosing(WindowEvent we) {
			requestExit();
		}

		public void keyPressed(KeyEvent e) {
			if (e.isControlDown()) {
				if (e.getKeyCode() == KeyEvent.VK_S) {
					saveText();
				} else if (e.getKeyCode() == KeyEvent.VK_R) {
					rejectText();
				} else if (e.getKeyCode() == KeyEvent.VK_N) {
					loadNext();
				} else if (e.getKeyCode() == KeyEvent.VK_P) {
					loadPrev();
				}
			}
		}

		public void keyTyped(KeyEvent e) {
			// TODO Auto-generated method stub
		}

		public void keyReleased(KeyEvent e) {
			// TODO Auto-generated method stub
		}
	}
}
