/*
 * @(#)DomconOzPluggingOutputCodec.java created 03.02.2006
 * 
 * Copyright (c) 2006 Alexander Koller
 *  
 */

package de.saar.chorus.domgraph.codec.plugging;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import de.saar.basic.StringTools;
import de.saar.chorus.domgraph.GlobalDomgraphProperties;
import de.saar.chorus.domgraph.codec.CodecMetadata;
import de.saar.chorus.domgraph.codec.CodecTools;
import de.saar.chorus.domgraph.codec.MalformedDomgraphException;
import de.saar.chorus.domgraph.codec.MultiOutputCodec;
import de.saar.chorus.domgraph.graph.DomEdge;
import de.saar.chorus.domgraph.graph.DomGraph;
import de.saar.chorus.domgraph.graph.NodeLabels;

/**
 * An output codec for pluggings in Oz syntax. This codec will print
 * an Oz list containing a term <code>plug(x y)</code> for each
 * dominance edge (x,y) in the solved form.<p>
 * 
 * An example output looks as follows:<br/>
 * {@code [plug(x y) plug(u v)]}
 * 
 * @author Alexander Koller
 *
 */

@CodecMetadata(name="plugging-oz", extension=".plug.oz")
public class DomconOzPluggingOutputCodec extends MultiOutputCodec {
    @Override
    public void encode(DomGraph graph, NodeLabels labels, Writer writer) 
    throws IOException, MalformedDomgraphException {
        Collection<DomEdge> domedges = graph.getAllDomEdges();
        List<String> edgeStrings = new ArrayList<String>();

        for( DomEdge edge : domedges ) {
            edgeStrings.add("plug(" + CodecTools.atomify(edge.getSrc())
                    + " " + CodecTools.atomify(edge.getTgt()) + ")");
        }
        
        writer.write("[" + StringTools.join(edgeStrings, " ") + "]\n");
        writer.flush();
    }

    @Override
    public void print_header(Writer writer) throws IOException {
        writer.write("%%  autogenerated by "
                + GlobalDomgraphProperties.getSystemName()
                + " (see " + GlobalDomgraphProperties.getHomepage()
                + " for details)\n");
    }

    @Override
    public void print_footer(Writer writer) throws IOException {
        writer.flush();
    }

    @Override
    public void print_start_list(Writer writer) throws IOException {
        writer.write("[");
    }

    @Override
    public void print_end_list(Writer writer) throws IOException {
        writer.write("]");
    }

    @Override
    public void print_list_separator(Writer writer) throws IOException {
    }

}
