/*
 *  File TreeSelector.java
 *
 *  Authors:
 *     Yannick Parmentier  <parmenti@sfs.uni-tuebingen.de>
 *     
 *  Copyright:
 *     Yannick Parmentier, 2007
 *
 *  Last modified:
 *     Di 16. Okt 11:13:42 CEST 2007
 *
 *  This file is part of the TuLiPA system
 *     http://www.sfb441.uni-tuebingen.de/emmy-noether-kallmeyer/tulipa
 *
 *  TuLiPA is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TuLiPA is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package de.tuebingen.anchoring;

import java.util.*;

import de.tuebingen.disambiguate.*;
import de.tuebingen.lexicon.*;
import de.tuebingen.tag.*;
import de.tuebingen.tokenizer.*;

public class TreeSelector {
	
	private boolean                     verbose;
	private List<Word>                   tokens;  // tokens (as Words)
	private List<Tuple>               anctuples;  // anchored tuples
	private Map<String, TagTree>       treeHash;  // dictionary to retrieve anchored trees at forest processing time
	private Map<String, List<String>> tupleHash;  // dictionary mapping a tuple id with tree ids
	private List<PolarizedToken>        ptokens;  // polarized tokens (for lexical disambiguation)
	private List<String>               lexNodes;  // lexical items in the trees (for polarity computation)
	private Map<String, List<String>>coancNodes;  // coanchors in the trees (for polarity computation)
	private Map<String, Integer>      ambiguity;
	

	public TreeSelector(List<Word> w, boolean v){
		verbose = v;
		tokens  = w;
		anctuples = new LinkedList<Tuple>();         // the anchored  tuples
		treeHash  = new Hashtable<String, TagTree>();// the trees dictionary
		tupleHash = new HashMap<String, List<String>>();
		ptokens   = new LinkedList<PolarizedToken>();
		lexNodes  = new LinkedList<String>();
		coancNodes= new HashMap<String, List<String>>();
		ambiguity = new HashMap<String,Integer>();
	}

	public void retrieve(Map<String, List<MorphEntry>> lm, Map<String, List<Lemma>> ll, Map<String, List<Tuple>> lt, List<String> slabels) {
		// lm is a mapping between a morph name and morph entries
		// ll is a mapping between a lemma name and lemma entries
		// lt is a mapping between a tuple name and tuple entries
		// slabels is where to store the semantic labels
		for (int i = 0; i < tokens.size(); i++){
			String s = tokens.get(i).getWord();
			//System.err.println("TreeSelector.retrieve --- s:\n" + s);
			ambiguity.put(s, 0); //init
			PolarizedToken ptk = new PolarizedToken(s, tokens.get(i).getEnd());
			if (lm.containsKey(s)) {
				List<MorphEntry> lme = lm.get(s);
				//System.err.println("TreeSelector.retrieve --- lme:\n" + lme);
				for (int j = 0 ; j < lme.size() ; j++){				
					retrieveLemma(ptk, new InstantiatedMorph(lme.get(j), tokens.get(i)), ll, lt, slabels);
				}
			} else {
				if (verbose) {System.err.println("Unknown token (not in the morph lexicon): "+s);}
			}
			ptokens.add(ptk);
		}
	}
	
	public void retrieveLemma(PolarizedToken ptk, InstantiatedMorph m, Map<String, List<Lemma>> ll, Map<String, List<Tuple>> lt, List<String> slabels){
		// m is a morph entry coupled with a lexical item from the input sentence
		// ll is a mapping between a lemma name and lemma entries
		// lt is a mapping between a tuple name and tuple entries
		
		// if the tokens have POS tags, we check these as well:
		// lemcats is a mapping Lemma,Cat for the current token
		Map<String, String> lemcats = m.getInToken().getTagAsAMap();

		// for each reference to a lemma included in the morph entry:
		List<Lemmaref> lm = m.getLemmarefs();
		for (int k = 0 ; k < lm.size() ; k++) {
			// we retrieve the name and cat of the lemma reference
			String lem = lm.get(k).getName();
			String cat = lm.get(k).getCat();
			// we check that either there is no POS-tagger
			// or that both the lemma  and the cat of the MorphEntry match the tagged lemma and cat
			if (lemcats == null || (lemcats.containsKey(lem) && lemcats.get(lem).equals(cat))) {
				if (ll.containsKey(lem)) {
					// we retrieve the lemmas matching the reference name
					List<Lemma> listlemma = ll.get(lem);
					for (int l = 0 ; l < listlemma.size() ; l++){
						// if both the lemma and the cat match the reference, we instantiate the lemma entry
						if (listlemma.get(l).getCat().equals(cat)){
							PolarizedLemma plm = new PolarizedLemma(listlemma.get(l).getName());
							InstantiatedLemma il = new InstantiatedLemma(listlemma.get(l), m, lm.get(k));
							retrieveTuples(plm, il, lt, slabels);
							updateCoAnchors(listlemma.get(l));
							lexNodes.addAll(plm.getLexicals());
							ptk.addLemma(plm);
						} else {
							if (verbose) 
								System.err.println("Rejected " + listlemma.get(l).toString() + " expected cat: " + cat);
						}
					}
				} else {
					if (verbose) 
						System.err.println("Unknown lemma (not in the lemma lexicon): "+lem);
				}
			} else {
				if (verbose)
					System.err.println("*** Lemma \"" + lem + "\" and category \"" + cat + "\" ignored.");
			}
		}
	}
	
	public void retrieveTuples(PolarizedLemma plm, InstantiatedLemma il, Map<String, List<Tuple>> lt, List<String> slabels) {
		// il is a lemma entry coupled with the lemma reference from the input morph
		// lt is a mapping between a tuple name and tuple entries
		// slabels is where to store the semantic labels
		
		int cpt_tmp = 0; // for counting ambiguity
		
		// for each anchoring scheme defined in the lemma entry
		List<Anchor> la  = il.getAnchors();
		for (int k = 0 ; k < la.size() ; k++) {
			// we retrieve the tuple name the scheme contains
			String family = la.get(k).getTree_id();
			if (lt.containsKey(family)) {
				// for each matching tuple
				for (int l = 0 ; l < lt.get(family).size() ; l++){
					cpt_tmp++;
					TagTree head = lt.get(family).get(l).getHead();
					// if the tuple's head's category match the anchoring scheme, we keep it for anchoring
					if (il.getCat().equals(((TagNode) head.getAnchor()).getCategory())) {
						InstantiatedTuple it = new InstantiatedTuple(la.get(k), lt.get(family).get(l), il);
						try {
							InstantiatedTuple x = this.anchor(plm, it, l, slabels);
							anctuples.add(x);
						} catch (AnchoringException e) {
							System.err.println("Tuple non-anchored: " + it.getId());
						}						
					} else {
						if (verbose) {System.err.println("Rejected " + head.getId() + " (categories do not match)");}
					}
				}				
			} else {
				if (verbose) {System.err.println("Unknown family (not in the tuple lexicon): "+family);}
			}
		}
		// ambiguity computation:
		String word = il.getLexItem().getLex();
		//System.err.println("Ambig for word " + word + " : " + cpt_tmp);
		int amb = ambiguity.get(word);
		amb += cpt_tmp;
		ambiguity.put(word, amb);
	}
	
	/**
	 * Method used to update the NameFactory with the semantic labels name
	 * prior to traverse the instantiated tree (c.f. feature sharing)
	 */
	public List<String> processSemLabels(List<SemLit> lsl, NameFactory nf) {
		List<String> slabels = new LinkedList<String>();
		for( SemLit sl: lsl ) {
			if (sl instanceof SemPred) {
				Value label = ((SemPred) sl).getLabel();
				// NB: labels have to be values (i.e. either constants or variables !)
				String newLabel = null;
				if (label.getSVal() != null) 
					newLabel = nf.getName(label.getSVal());
				else if (label.getVarVal() != null)
					newLabel = nf.getName(label.getVarVal());
				// thus the name factory is initialized with the labels to rename
				if (!slabels.contains(newLabel))
					slabels.add(newLabel);
			}
		}
		return slabels;
	}
	
	public InstantiatedTuple anchor(PolarizedLemma plm, InstantiatedTuple t, int i, List<String> slabels) throws AnchoringException {
		/**
		 * Method processing each instantiated tuples of the selector
		 * in order to anchor it (unification between FS and lexical anchoring of the head)
		 */
		if (verbose)
			System.err.println("\nAnchoring ... " + t.getId() + "\n");
		NameFactory nf = new NameFactory();
		
		// First, we get the semantic labels and rename them using the name factory
		List<SemLit> lsl = t.getHead().getSem();
		List<String> semlabels = this.processSemLabels(lsl, nf);

		// we retrieve the tuple's head (and use a duplicate!)
		TagTree hd = t.getHead(); 
		TagTree tt = new TagTree(hd, nf);
			
		// we build unique tree and tuple identifiers
		String newTreeId  = tt.getId()+"--"+(t.getLemma().getLexItem().getLex()) + (t.getLemma().getLexItem().getInToken().getEnd()) + "--" + i;
		String newTupleId = tt.getTupleId()+"--"+(t.getLemma().getLexItem().getLex()) + (t.getLemma().getLexItem().getInToken().getEnd()) + "--" + i;

		tt.setId(nf.getName(newTreeId));
		tt.setOriginalId(newTreeId);
		tt.setTupleId(nf.getName(newTupleId));
		tt.setOriginalTupleId(newTupleId);
		tt.setIsHead(true);
		tt.setTupleAncPos((t.getLemma().getLexItem().getLex()) + (t.getLemma().getLexItem().getInToken().getEnd()));
		
		// we retrieve the tuple's arguments (and use duplicates!)
		List<TagTree> args = t.getArguments();			
		LinkedList<TagTree> tl = null;
		if (args != null) {
			tl = new LinkedList<TagTree>();
			for (int k = 0 ; k < args.size() ; k++) {
				TagTree targ = new TagTree(args.get(k), nf);
				// we build unique tree and tuple identifiers for arguments
				targ.setId(nf.getName(targ.getId()+"--"+(t.getLemma().getLexItem().getLex()) + (t.getLemma().getLexItem().getInToken().getEnd())  + "--" + i));
				targ.setOriginalId(args.get(k).getId()+"--"+(t.getLemma().getLexItem().getLex()) + (t.getLemma().getLexItem().getInToken().getEnd())  + "--" + i);
				targ.setTupleId(nf.getName(newTupleId));
				targ.setOriginalTupleId(newTupleId);
				targ.setTupleAncPos(tt.getTupleAncPos());
				tl.add(targ);
				// we do not update the tree dictionary yet since we do not know whether the anchoring will succeed
				//treeHash.put(targ.getId(), targ);
			}
		}
		//System.err.println("Tree "+hd.getId());
			
		Fs ancfs   = t.getAnchor().getFilter();
		Fs iface   = tt.getIface();
		// the default size of the environment is 10 (trade-off) 
		Environment env = new Environment(10);
			
		// a) unify the head tree interface with the filter 
		Fs uniface = null;
		if (ancfs != null && iface != null) {
			try {
				uniface = Fs.unify(ancfs, iface, env);
			} catch (UnifyException e){
				System.err.println("Interface unification failed on tree "+ tt.getOriginalId() + " for filter " + ancfs.toString());
				System.err.println(e);
				throw new AnchoringException(); // we withdraw the current anchoring
			}
		} else if (ancfs == null) {
			uniface = iface;
		} else {
			uniface = ancfs;
		}
		tt.setIface(uniface);
		//System.err.println("\n (after iface processing)");
		//System.err.println(env.toString());
			
		// b) tree traversal to apply node equations
		//    node equations concern both the head and argument trees	
		//    if an equation fails, the tuple is discarded (not anchored)
		List<Equation> leq = t.getAnchor().getEquations();
		for (int j = 0 ; j < leq.size() ; j++) {
			boolean equationOk = false;
			Equation eq = leq.get(j);
			try {
				equationOk |= tt.solveEquation(tt.getRoot(), eq, env);
			} catch (UnifyException e) {
				System.err.println("Equation solving failed on tree "+ tt.getOriginalId() + " for equation " + eq.toString());
				System.err.println(e);
				throw new AnchoringException(); // we withdraw the current anchoring
			}
			if (tl != null) {	
				for (int k = 0 ; k < tl.size() ; k++) {
					TagTree targ = tl.get(k);
					try {
						equationOk |= targ.solveEquation(targ.getRoot(), eq, env);
					} catch (UnifyException e) {
						System.err.println("Equation solving failed on tree "+ tt.getOriginalId() + " for equation " + eq.toString());
						System.err.println(e);
						throw new AnchoringException(); // we withdraw the current anchoring
					}
				}
			}
			if (!equationOk) {
				System.err.println("Equation solving failed on tree "+ tt.getOriginalId() + " for equation " + eq.toString() + " (node not found)");
				throw new AnchoringException(); // we withdraw the current anchoring
			}
		}
		//System.err.println(" (after equation processing)");
		//System.err.println(env.toString());
			
		// c) add the new lexical item under the anchor node
		//    comes with unification of morph features
		try {
			tt.anchor(t.getLemma().getLexItem().getInToken(), t.getLemma().getLref(), env);
		} catch (UnifyException e) {
			System.err.println("Anchoring failed on tree "+tt.getOriginalId()+" for lexical item " + t.getLemma().getLexItem().getLex());
			System.err.println(e);
			throw new AnchoringException(); // we withdraw the current anchoring
		}
		//System.err.println(" (after anchoring)");
		//System.err.println(env.toString());
			
		// d) we perform the co-anchor equations
		List<CoAnchor> lca = t.getAnchor().getCoanchors();
		if (lca != null) {
			for(int j = 0 ; j < lca.size() ; j++) {
				boolean coaOk = false;
				// try to co-anchor the head
				int k = 0;
				coaOk = tt.coanchor(tt.getRoot(), lca.get(j));
				// try to co-anchor the arguments
				if (tl != null) {
					while (k < tl.size() && !coaOk) {
						coaOk = tl.get(k).coanchor(tl.get(k).getRoot(), lca.get(j));
						k++;
					}
				}
				if (!coaOk) {
					// if the co-anchor has not been found, do not use the tuple
					System.err.println("Co-anchor not found : " + lca.get(j).getNode_id() + " (tuple " + tt.getOriginalId() + " discarded).");
					throw new AnchoringException();
				}
			}
		}
		// e) we instantiate the semantic information
		//    we unify the tree interface with the semantic class call from the lemma, 
		//    provided the semantic class is part of the trace!
		//    This gives values to instantiate the semantic information of the tree
		List<LexSem> lemmaSem = t.getAnchor().getSemantics();
		List<String> treeTrace= tt.getTrace();
		List<SemLit> treeSem  = tt.getSem();
		if (lemmaSem != null) {
			for(int k = 0 ; k < lemmaSem.size() ; k++) {
				if (treeTrace.contains(lemmaSem.get(k).getSemclass())) {
					// if the called semantic class has been used in the tree (cf trace)
					// we unify the interface and semantic arguments
					Fs semFs  = new Fs(lemmaSem.get(k).getArgs());
					try {
						Fs.unify(semFs, tt.getIface(), env);
						// the environment now contains the bindings for semantic variables
						// we can update the tree semantics
						for(int ksem = 0 ; ksem < treeSem.size() ; ksem++) {
							treeSem.get(ksem).update(env, false);
						}
					} catch (UnifyException e) {
						System.err.println("Semantic features unification failed on tree "+ hd.getOriginalId() + " for semantic class " + lemmaSem.get(k).toString());
						System.err.println(e);
						throw new AnchoringException(); // we withdraw the current anchoring
					}
				}
			}
		}
		//System.err.println(" (after semantic anchoring)");
		//System.err.println(env.toString());
			
		// e') we update all the FS in the trees according to the env state
		try {
			tt.updateFS(tt.getRoot(), env, false);
		} catch (UnifyException e) {
			System.err.println("FS update failed on tree "+tt.getOriginalId());
			System.err.println(e);
			throw new AnchoringException(); // we withdraw the current anchoring
		}
		if (tl != null) {
			for (int j = 0 ; j < tl.size() ; j++) {
				try {
					tl.get(j).updateFS(tl.get(j).getRoot(), env, false);
				} catch (UnifyException e) {
					System.err.println("FS update failed on tree "+tl.get(j).getOriginalId());
					System.err.println(e);
					throw new AnchoringException(); // we withdraw the current anchoring
				}
			}
		}
		//System.err.println(" (after FS updating)");
		//System.err.println(env.toString());
			
		// f) We store the result
		InstantiatedTuple x = new InstantiatedTuple();
		List<String> xTrees = new LinkedList<String>();
		String tupleId = nf.getName(t.getId());
		x.setId(tupleId);
		x.setOriginalId(t.getId());
		x.setFamily(t.getFamily());
		x.setLemma(t.getLemma());
		x.setSemLabels(semlabels);
		slabels.addAll(semlabels);
		// we initialize the tuple's polarities
		Polarities p = new Polarities();
		PolarizedTuple ptl = new PolarizedTuple(x.getId());
		ptl.setOriginalID(x.getOriginalId());
		// we add the head's polarities
		tt.getPolarities(p);
		x.setHead(tt);
		ptl.addLexicals(tt.getLexItems());
		// we update the tree dictionary
		//-------------------------------
		treeHash.put(tt.getId(), tt);
		// and the tuple:
		xTrees.add(tt.getId());
		//System.err.println("Added " + tt.getOriginalId());
		//System.err.println(tt.getRoot().toString());
		if (tl != null) {
			for(int l = 0 ; l < tl.size() ; l++) {
				treeHash.put(tl.get(l).getId(), tl.get(l));
				xTrees.add(tl.get(l).getId());
				tl.get(l).getPolarities(p);
				ptl.addLexicals(tl.get(l).getLexItems());
			}
			x.setArguments(tl);
		}
		//-------------------------------
		tupleHash.put(x.getId(), xTrees);
		ptl.setPol(p);
		plm.addTuple(ptl);
		return x;
	}
	
	public void store(Map<String, List<Tuple>> grammar){
		/**
		 * Method used when no anchoring is needed, the tree selector
		 * then only stores the trees to the dictionary and lists.
		 */
		Set<String> keys = grammar.keySet();
		Iterator<String> it = keys.iterator();
		while(it.hasNext()) {
			String tupleId = it.next();
			List<Tuple> tup = grammar.get(tupleId);
			for(int i = 0 ; i < tup.size() ; i++){
				Tuple tu = tup.get(i);
				anctuples.add(tu);
				tu.getHead().setOriginalId(tu.getHead().getId());
				treeHash.put(tu.getHead().getId(), tu.getHead());
				List<TagTree> args = tu.getArguments();
				if (args != null) {
					for (int j = 0 ; j < args.size() ; j++){
						TagTree ttree = args.get(j);
						ttree.setOriginalId(ttree.getId());
						treeHash.put(ttree.getId(), ttree);
					}
				}
			}
		}
	}

	public boolean isVerbose() {
		return verbose;
	}

	public void setVerbose(boolean verbose) {
		this.verbose = verbose;
	}

	public List<Word> getTokens() {
		return tokens;
	}

	public void setTokens(List<Word> tokens) {
		this.tokens = tokens;
	}

	public List<Tuple> getAnctuples() {
		return anctuples;
	}

	public void setAnctuples(List<Tuple> anctuples) {
		this.anctuples = anctuples;
	}
	
	public Map<String, TagTree> getTreeHash() {
		return treeHash;
	}

	public Map<String, List<String>> getTupleHash() {
		return tupleHash;
	}

	public List<PolarizedToken> getPtokens() {
		return ptokens;
	}

	public List<String> getLexNodes() {
		return lexNodes;
	}

	public Map<String, List<String>> getCoancNodes() {
		return coancNodes;
	}
	
	public String toString(){
		String res = "";
		res += "\n Retrieved tuples : \n\t";
		for (int i = 0 ; i < anctuples.size() ; i++) {
			InstantiatedTuple t = (InstantiatedTuple) anctuples.get(i); 
			res += t.getOriginalId() + "[" + t.getId() +"]("+t.getLemma().getName()+"-"+t.getLemma().getLexItem().getLex()+") ";
		}
		res += "\n Anchored tuples : \n";
		for (int i = 0 ; i < anctuples.size() ; i++) {
			Tuple t =  anctuples.get(i);
			res += "\n Tuple "+(i+1)+": \n";
			TagTree anchored = t.getHead();
			res += anchored.toString("  ");
			if (t.getArguments() != null) {
				for (int j = 0 ; j < t.getArguments().size() ; j++) {
					TagTree arg = t.getArguments().get(j);
					res+= arg.toString("  ");
				}
			}
		}
		return res;
	}

	public void addCoanchor(String word, String cat) {
		List<String> cats = null;
		if (coancNodes.containsKey(word))
			cats = coancNodes.get(word);
		else
			cats = new LinkedList<String>();
		if (!cats.contains(cat))
			cats.add(cat);
		coancNodes.put(word, cats);
	}
	
	public void updateCoAnchors(Lemma l) {
		List<CoAnchor> lca = l.getCoAnchors();
		for(CoAnchor a : lca) {
			List<String> words = a.getLex();
			for(String s : words) {
				this.addCoanchor(s, a.getCat());
			}
		}
	}
	
	public long getambig() {
		long res = 1;
		Iterator<String> it = ambiguity.keySet().iterator();
		while (it.hasNext()) {
			String tok = it.next();
			int num = ambiguity.get(tok);
			//System.err.println(tok + " , " + num);
			if (num != 0) //for lexical nodes
				res *= num;
		}
		return res;
	}
	
}
