package parser;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import trees.SwigraTreeNode;
import utility.Utility;

public class SwigraTree implements Serializable, Comparable<SwigraTree> {

	private static final long serialVersionUID = -4730103474372470486L;
	private String filepath;
	private Map<Integer, SwigraTreeNode> nodes;
	public boolean fromDep = false;
	//public List<String> annotators;

	public SwigraTree(Map<Integer, SwigraTreeNode> nodes/*, List<String> annotators*/, String filepath, boolean full, boolean fromDep)
			throws HeadlessException {
		this.nodes = nodes;
		this.filepath = filepath;
		this.fromDep = fromDep;
		for (SwigraTreeNode n: this.nodes.values()) {
			n.fromDep = this.fromDep;
		}
		//this.annotators = annotators;
		this.reduce(0);
		if (full && !this.fromDep) {
			this.markHeads(0);
		}
		this.linkTree(0);
		if (full) {
			this.parsePOS(this.getRoot());
		}
	}

	@Override
	public int compareTo(SwigraTree other) {
		return this.filepath.compareTo(other.filepath);
	}
	
	@Override
	public boolean equals(Object other) {
		if (other.getClass() != this.getClass()) {
			return false;
		} else {
			return this.fullTexTree().contentEquals(((SwigraTree) other).fullTexTree());
		}
	}
	
	@Override
	public int hashCode() {
		return this.fullTexTree().hashCode();
	}
	
	public String texTree() {
		return "\\synttree\n" + this.texTree("", this.getRoot(), false).replace("_", "\\_")
				+ "\\newpage\n";
	}
	
	public String fullTexTree() {
		return "\\synttree\n" + this.texTree("", this.getRoot(), true).replace("_", "\\_")
				+ "\\newpage\n";
	}

	private String texTree(String tab, SwigraTreeNode node, boolean expanded) {
		boolean exp = expanded || this.containsError(node);
		StringBuilder tex = new StringBuilder(tab);
		tex.append("[");
		if (node.error) {
			tex.append("{\\color{red}");
		}
		if (node.isHead) {
			tex.append("\\textbf{");
		}
		tex.append(node.fs.containsKey(Feature.TFW) ? "!" : "");
		tex.append(node.fs.containsKey(Feature.JAKO_FL) ? "?" : "");
		tex.append((node.tag == SwigraTag.LEAF ? "" : node.tag) + node.orth);
		if (node.isHead) {
			tex.append("}");
		}
		if (node.error) {
			tex.append("}");
		}
		tex.append("\n");
		if (exp) {
			for (SwigraTreeNode child : node.children) {
				//System.out.println(child.tag);
				tex.append(this.texTree(tab + "  ", child, expanded || node.error));
			}
		}
		tex.append(tab);
		tex.append("]\n");
		return tex.toString();
	}

	public String getFilePath() {
		return this.filepath;
	}

	public SwigraTreeNode getRoot() {
		return this.nodes.get(0);
	}

	public boolean containsError() {
		return this.containsError(this.getRoot());
	}

	private boolean containsError(SwigraTreeNode node) {
		if (node.error) {
			return true;
		} else {
			for (SwigraTreeNode child : node.children) {
				if (this.containsError(child)) {
					return true;
				}
			}
			return false;
		}
	}
	
	public List<SwigraTreeNode> terminalNodes() {
		return this.terminalNodes(this.getRoot());
	}

	private List<SwigraTreeNode> terminalNodes(SwigraTreeNode node) {
		List<SwigraTreeNode> tns = new LinkedList<SwigraTreeNode>();
		if (node.isTerminal()) {
			tns.add(node);	
		}
		else {
			for (SwigraTreeNode child : node.children) {
				tns.addAll(this.terminalNodes(child));
			}
		}
		return tns;
	}
	
	public List<String> words() {
		return this.getRoot().orths();
	}

	public String sentence() {
		return this.getRoot().sentence();
	}

	public List<String> POSs() {
		return this.getRoot().POSs();
	}
	
	public List<SwigraTreeNode> errorNodes() {
		return this.getRoot().errorNodes();
	}
	
	public List<SwigraTreeNode> markedNodes() {
		return this.getRoot().markedNodes();
	}

	private void reduce(Integer nodeId) {
		SwigraTreeNode node = this.nodes.get(nodeId);
		for (int childNo = 0; childNo < node.childrenIds.size(); ) {
			if (this.nodes.get(node.childrenIds.get(childNo)).tag == SwigraTag.FW
					|| this.nodes.get(node.childrenIds.get(childNo)).tag == SwigraTag.FL) {
				SwigraTreeNode oldChild = this.nodes.get(node.childrenIds
						.get(childNo));
				node.childrenIds.remove(childNo);
				for (Integer headChild : oldChild.headChildren) {
					SwigraTreeNode newChild = this.nodes.get(headChild);
					if (oldChild.tag == SwigraTag.FW) {
						newChild.fs.put(Feature.TFW, oldChild.fs.get(Feature.TFW));
					}
					if (oldChild.tag == SwigraTag.FL) {
						newChild.fs.put(Feature.JAKO_FL, "fl");
					}
					node.childrenIds.add(childNo++, newChild.id);
				}
			} else {
				++childNo;
			}
		}
		for (Integer childId : node.childrenIds) {
			this.reduce(childId);
		}
	}

	private void markHeads(Integer nodeId) throws HeadlessException {
		SwigraTreeNode node = this.nodes.get(nodeId);
		boolean headFound = false;
		if (node.headChildren.size() > 1) {
			//System.out.println(node.headChildren.size() + "-głowy smok!     " + node.tag);
			//System.out.println("    " + this.filepath);
		}
		for (Integer childId : node.childrenIds) {
			if (node.headChildren.contains(childId)) {
				this.nodes.get(childId).isHead = true;
				headFound = true;
			}
			this.markHeads(childId);
		}
		if (node.tag != SwigraTag.LEAF && !headFound) {
			if (node.childrenIds.size() == 1) {
				node.headChildren.add(node.childrenIds.get(0));
				this.nodes.get(node.childrenIds.get(0)).isHead = true;
			} else {
				System.out.println("NO HEAD - " + node.tag + " "
						+ this.filepath);
				throw new HeadlessException();
			}
		}
	}

	private void linkTree(Integer nodeId) {
		SwigraTreeNode node = this.nodes.get(nodeId);
		for (Integer childId : node.childrenIds) {
			((List<SwigraTreeNode>) node.children).add(this.nodes.get(childId));
			this.linkTree(childId);
		}

	}

	private void parsePOS(SwigraTreeNode node) {
		if (node.tag == SwigraTag.LEAF || this.fromDep) {
			String[] posTags = node.pos.split(":");
			String pos = posTags[0];
			node.fs.put(Feature.POS, pos);
			for (int i = 1; i < posTags.length; ++i) {
				posTags[i] = Utility.translatePosTag(posTags[i]);
			}
			/*
			 * interp, padj, padv, psubst
			 */
			if (pos.contentEquals("subst") || pos.contentEquals("depr") || pos.contentEquals("psubst")) {
				// liczba przypadek rodzaj
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.PRZYPADEK, posTags[2]);
				node.fs.put(Feature.RODZAJ, posTags[3]);
			}
			if (pos.contentEquals("num") || pos.contentEquals("numcol")) {
				// liczba przypadek rodzaj akomod
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.PRZYPADEK, posTags[2]);
				node.fs.put(Feature.RODZAJ, posTags[3]);
				node.fs.put(Feature.AKOM, posTags[4]);
			}
			if (pos.contentEquals("adj") || pos.contentEquals("padj")) {
				// liczba przypadek rodzaj stopien
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.PRZYPADEK, posTags[2]);
				node.fs.put(Feature.RODZAJ, posTags[3]);
				node.fs.put(Feature.STOPIEN, posTags[4]);
			}
			if (pos.contentEquals("adja") || pos.contentEquals("adjp")
					|| pos.contentEquals("adjc") || pos.contentEquals("pred")
					|| pos.contentEquals("conj") || pos.contentEquals("comp")
					|| pos.contentEquals("qub")) {
				//
			}
			if (pos.contentEquals("adv") || pos.contentEquals("padv")) {
				// stopien
				if (posTags.length > 1) {
					node.fs.put(Feature.STOPIEN, posTags[1]);
				}
			}
			if (pos.contentEquals("ppron12")) {
				// liczba przypadek rodzaj osoba akcent
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.PRZYPADEK, posTags[2]);
				node.fs.put(Feature.RODZAJ, posTags[3]);
				node.fs.put(Feature.OSOBA, posTags[4]);
				if (posTags.length > 5) {
					node.fs.put(Feature.AKCENT, posTags[5]);
				}
			}
			if (pos.contentEquals("ppron3")) {
				// liczba przypadek rodzaj osoba akcent poprzyim
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.PRZYPADEK, posTags[2]);
				node.fs.put(Feature.RODZAJ, posTags[3]);
				node.fs.put(Feature.OSOBA, posTags[4]);
				node.fs.put(Feature.AKCENT, posTags[5]);
				node.fs.put(Feature.POPRZYIMK, posTags[6]);
			}
			if (pos.contentEquals("siebie") || pos.contentEquals("prep")) {
				// przypadek wokal
				node.fs.put(Feature.PRZYPADEK, posTags[1]);
				if (posTags.length > 2) {
					node.fs.put(Feature.WOKAL, posTags[2]);
				}
			}
			if (pos.contentEquals("fin") || pos.contentEquals("bedzie")) {
				// liczba osoba aspekt
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.OSOBA, posTags[2]);
				node.fs.put(Feature.ASPEKT, posTags[3]);
			}
			if (pos.contentEquals("aglt")) {
				// liczba osoba aspekt wokal
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.OSOBA, posTags[2]);
				node.fs.put(Feature.ASPEKT, posTags[3]);
				node.fs.put(Feature.WOKAL, posTags[4]);
			}
			if (pos.contentEquals("praet")) {
				// liczba rodzaj aspekt aglutyn
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.RODZAJ, posTags[2]);
				node.fs.put(Feature.ASPEKT, posTags[3]);
				if (posTags.length > 4) {
					node.fs.put(Feature.AGLUTYN, posTags[4]);
				}
			}
			if (pos.contentEquals("impt")) {
				// liczba osoba aspekt
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.OSOBA, posTags[2]);
				node.fs.put(Feature.ASPEKT, posTags[3]);
			}
			if (pos.contentEquals("imps") || pos.contentEquals("inf")
					|| pos.contentEquals("pcon") || pos.contentEquals("pant")) {
				// aspekt
				node.fs.put(Feature.ASPEKT, posTags[1]);
			}
			if (pos.contentEquals("ger") || pos.contentEquals("pact")
					|| pos.contentEquals("ppas")) {
				// liczba przypadek rodzaj aspekt zaneg
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.PRZYPADEK, posTags[2]);
				node.fs.put(Feature.RODZAJ, posTags[3]);
				node.fs.put(Feature.ASPEKT, posTags[4]);
				node.fs.put(Feature.NEG, posTags[5]);
			}
			if (pos.contentEquals("winien")) {
				// liczba rodzaj aspekt
				node.fs.put(Feature.LICZBA, posTags[1]);
				node.fs.put(Feature.RODZAJ, posTags[2]);
				node.fs.put(Feature.ASPEKT, posTags[3]);
			}
			if (pos.contentEquals("brev")) {
				// kropk
				node.fs.put(Feature.KROPK, posTags[1]);
			}
		} else {
			for (Object child : node.children) {
				this.parsePOS((SwigraTreeNode) child);
			}
		}
	}
	
	public boolean containsFnoFlicz() {
		return this.containsFnoFlicz(this.getRoot());
	}
	
	private boolean containsFnoFlicz(SwigraTreeNode node) {
		if (node.rule.contentEquals("nol1") || node.rule.contentEquals("nol2") || node.rule.contentEquals("nol3")) {
			return true;
		}
		for (SwigraTreeNode child : node.children) {
			if (this.containsFnoFlicz(child)) {
				return true;
			}
		}
		return false;
	}
	
	/*public STSGTreeNode makeSTSGTree() {
		return this.makeSTSGTree(this.getRoot());
	}
	
	private STSGTreeNode makeSTSGTree(SwigraTreeNode node) {
		STSGTreeNode n = new STSGTreeNode(node);
		for (SwigraTreeNode child : node.children) {
			n.children.add(this.makeSTSGTree(child));
		}
		return n;
	}
	
	public List<String> STSGPOSs() {
		List<String> pos = new LinkedList<String>();
		this.STSGPOSs(this.getRoot(), pos);
		return pos;
	}
	
	private void STSGPOSs(SwigraTreeNode node, List<String> pos) {
		if (node.isTerminal()) {
			String tag;
			tag = node.fs.get(Feature.POS);
			for (Feature f : STSGTreeNode.terminalFeatures) {
				if (node.fs.containsKey(f)) {
					tag += "*" + node.fs.get(f);
				}
			}
			pos.add(tag);
		}
		for (SwigraTreeNode child : node.children) {
			this.STSGPOSs(child, pos);
		}
	}*/

}
