package tag;

import java.io.File;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Document;

import de.saar.chorus.dtool.DTool;
import de.tuebingen.anchoring.NameFactory;
import de.tuebingen.anchoring.TreeSelector;
import de.tuebingen.converter.GrammarConvertor;
import de.tuebingen.dependency.DependencyDOMbuilder;
import de.tuebingen.dependency.DependencyExtractor;
import de.tuebingen.disambiguate.ComputeSubGrammar;
import de.tuebingen.disambiguate.PolarityAutomaton;
import de.tuebingen.disambiguate.PolarizedToken;
import de.tuebingen.expander.DOMderivationBuilder;
import de.tuebingen.forest.ProduceDOM;
import de.tuebingen.gui.DerivedTreeViewer;
import de.tuebingen.parser.ForestExtractor;
import de.tuebingen.parser.ForestExtractorFactory;
import de.tuebingen.parser.RCGParser;
import de.tuebingen.parser.RCGParserBoullier2;
import de.tuebingen.parser.simple.SimpleRCGParserEarley;
import de.tuebingen.parserconstraints.RCGParserConstraintEarley;
import de.tuebingen.rcg.RCG;
import de.tuebingen.rcg.RCGDOMbuilder;
import de.tuebingen.tag.TagTree;
import de.tuebingen.tag.Tuple;
import de.tuebingen.tagger.ExternalTagger;
import de.tuebingen.tagger.TaggerException;
import de.tuebingen.tokenizer.Tokenizer;
import de.tuebingen.tokenizer.Word;
import de.tuebingen.tree.Grammar;
import de.tuebingen.ui.CommandLineOptions;
import de.tuebingen.ui.Interface;
import de.tuebingen.util.CollectionUtilities;
import de.tuebingen.util.XMLUtilities;

public class AlternativeInterface extends Interface {
	
	public static boolean parseSentence(CommandLineOptions op, Grammar g, String sentence, List<Word> tokens) throws Exception {
		boolean res = false;
	    long totalTime = 0;
		boolean verbose = op.check("v");
		boolean noUtool = op.check("n");
		boolean needsAnchoring = g.needsAnchoring();

		String outputfile = "";
		if (op.check("o")) { outputfile = op.getVal("o"); }
		else { outputfile = "stdout"; }
		String axiom = "v"; // default axiom's value is v
		if (op.check("a")) {axiom = op.getVal("a");}
		
		List<String> slabels = new LinkedList<String>();
		
		// 4. Load the tokenizer
		Tokenizer tok = loadTokenizer(op);
		tok.setSentence(sentence);
		if (verbose) {
			System.err.println("Tokenized sentence: " + tokens.toString());
		}
		List<String> toksentence = Tokenizer.tok2string(tokens);
		//System.err.println("\t@@ Length " + toksentence.size());
		
		/* ******** external POS tagging ************/
		ExternalTagger tagger = new ExternalTagger();
		File taggerExec = op.check("t") ? new File(op.getVal("t")) : null; 
		tagger.setExec(taggerExec);
		tagger.setParams("");
		try {
			 tagger.doTagging(tokens);
		} catch (TaggerException e) {
			System.err.println(" ********** Tagging Exception *********");
			System.err.println(e.toString());
		}
		//ExternalTagger.printPosToken(tokens);
		/* ******************************************/
		
		// 5. Lexical selection and Anchoring
		TreeSelector ts = new TreeSelector(tokens, verbose);
		List<List<Tuple>> subgrammars = null;
		
		if (needsAnchoring) {
			long ancTime = System.nanoTime();
			// 5-a. According to the tokens, we retrieve the pertinent morph entries
			// 5-b. According to the morph entries, we instantiate the pertinent lemmas
			// 5-c. According to the instantiated lemmas, we instantiate the pertinent tuples
			// 6. Tree anchoring
			ts.retrieve(g.getMorphEntries(), g.getLemmas(), g.getGrammar(), slabels);
			//System.err.println(ts.toString());
			//System.err.println(ts.getTupleHash());
			
			long anchoredTime = System.nanoTime() - ancTime;
			System.err.println("Grammar anchoring time: " + (anchoredTime)/(Math.pow(10, 9))+" sec.");
			if (verbose) 
				System.err.println("Anchoring results:\n" + ts.toString());
			totalTime += anchoredTime; 
			if (!op.check("nofiltering")) {
				//--------------------------------------------------------
				// before RCG conversion, we apply lexical disambiguation:
				//--------------------------------------------------------
				List<PolarizedToken> lptk = ts.getPtokens();
				if (verbose) {
					for(PolarizedToken ptk : lptk){
						System.err.println(ptk.toString());
					}
				}
				PolarityAutomaton pa = new PolarityAutomaton(toksentence, lptk, axiom, verbose, ts.getLexNodes(), ts.getCoancNodes());
				List<List<String>> tupleSets = pa.getPossibleTupleSets();
				subgrammars = ComputeSubGrammar.computeSubGrammar(verbose, tupleSets, ts.getTupleHash(), ts.getTreeHash()); 

				System.err.println("\t@@##Tree combinations before classical polarity filtering   : " + ts.getambig());
				System.err.println("\t@@##Tree combinations after classical polarity filtering   : " + CollectionUtilities.computeAmbig(tupleSets, toksentence) + "\n");
				
				if (verbose) {
					System.err.println("Valid tuple sets:\n" + tupleSets);
					//System.err.println("\nCorresponding sub-grammars:\n" + subgrammars);
				}
				//--------------------------------------------------------
			}
			else {
				System.err.println("\n\t@@##Tree combinations after left polarity filtering   : " + ts.getambig() + "\n");
			}
		} else {
			ts.store(g.getGrammar());
		}
		// Tree Selection results stored in specific variables to avoid 
		// keeping a pointer to the ts variable (and wasting memory)
		Map<String, TagTree> grammarDict = ts.getTreeHash();
		List<Tuple>       anchoredTuples = ts.getAnctuples();
		//For debugging:
		/*
		Iterator<String> its = grammarDict.keySet().iterator();
		while (its.hasNext()) {
			String k =  its.next();
			TagTree tagt = grammarDict.get(k);
			System.err.println(tagt.toString(""));
		}
		for(Tuple t : ts.getAnctuples()) {
			System.err.println(t);
		}
		*/
		
		// 7. RCG conversion
		int limit   = op.check("z") ? Integer.parseInt(op.getVal("z")) : -1;
		int k_limit = op.check("k") ? Integer.parseInt(op.getVal("k")) : -1;

		RCG rcggrammar = null;
		long startTime = System.nanoTime();	
		if (subgrammars != null) { // i.e. we used lexical disambiguation 
			rcggrammar = new RCG();
			for(int sI = 0 ; sI < subgrammars.size() ; sI++) {
				List<Tuple> ltuples = subgrammars.get(sI);
				//System.err.println("Converting sub-grammar " + sI + "...");
				GrammarConvertor gc = new GrammarConvertor(ltuples, verbose, toksentence, grammarDict, !needsAnchoring, k_limit, limit);
				gc.buildAllClauses(axiom);
				rcggrammar.addGrammar(gc.getRcggrammar(), grammarDict);
				//OptimizedGrammarConverter ogc = new OptimizedGrammarConverter(anchoredTuples, verbose, toksentence, grammarDict, !needsAnchoring);
				//ogc.buildAllClauses(axiom);				
				//rcggrammar.addGrammar(ogc.getRcggrammar(), grammarDict);
			}
		} else {
			GrammarConvertor gc = new GrammarConvertor(anchoredTuples, verbose, toksentence, grammarDict, !needsAnchoring, k_limit, limit);
			gc.buildAllClauses(axiom);
			//OptimizedGrammarConverter ogc = new OptimizedGrammarConverter(anchoredTuples, verbose, toksentence, grammarDict, !needsAnchoring);
			//ogc.buildAllClauses(axiom);
			rcggrammar = new RCG();
			rcggrammar.addGrammar(gc.getRcggrammar(), grammarDict);		
			//rcggrammar.addGrammar(ogc.getRcggrammar(), grammarDict);
		}
		long estimatedTime = System.nanoTime() - startTime;
		totalTime += estimatedTime;
		if (rcggrammar == null || rcggrammar.getStartPredicateLabel() == null) {
			//System.err.println("Grammar conversion failed. \nPlease check the value of the axiom.");
			throw new Exception("Polarity filtering / grammar conversion failed. \nPlease check the value of the axiom and the lexicon.");
		} else
			System.err.println("Grammar conversion time: "+(estimatedTime)/(Math.pow(10, 9))+" sec.");
		// for printing the RCG grammar computed
		// either pretty printed:
		//System.err.println(rcggrammar.toString(ts.getTreeHash())+"\n");
		// --------------------------------------------------------------
		// or not (i.e. the real grammar + id interpretations):
		/*
		//if (true) {
		if (verbose) {
			Iterator<String> itt = grammarDict.keySet().iterator();
			while(itt.hasNext()) {
				String ttree = itt.next();
				System.err.println("Tree " + ttree + "\t := " + grammarDict.get(ttree).getOriginalId() + " " + grammarDict.get(ttree).toString(""));
			}
			//System.err.println(rcggrammar.toString());
		}
		*/

		// 7'. RCG XML export (for DyALog)
		if (op.check("e")) {
			System.err.println(rcggrammar.toString(grammarDict));
			Document rcgd = RCGDOMbuilder.exportGrammar(rcggrammar, grammarDict);
			XMLUtilities.writeXML(rcgd, op.getVal("e"), "rcg.dtd,xml", true);
		}
		
		// 8. RCG parsing
		RCG rcgg = rcggrammar;
		
		RCGParser parser = null;
		if (op.check("q2") || op.check("earley")) {
			if (verbose)
				System.err.println("Using CSP earley parser");
			// full parser from Kallmeyer&Maier&Parmentier(2009), using constraint programming
			parser = new RCGParserConstraintEarley(rcgg);
			int verb = (verbose) ? 0 : -2;
			((RCGParserConstraintEarley) parser).setVerbose(verb);
			((RCGParserConstraintEarley) parser).setGDict(grammarDict);
		} else if (op.check("full")) {
			if (verbose)
				System.err.println("Using Top-Down parser from Boullier (2000)");
			parser = new RCGParserBoullier2(verbose, rcgg, grammarDict, rcgg.getCategories());
		}
		// simple RCG parser is the default		
		if (parser == null) {
			if (verbose)
				System.err.println("Using simple RCG parser");
			parser = new SimpleRCGParserEarley(rcgg);
		}

		// for printing the categories modifiable within the subgrammar:
		//System.err.println(rcgg.getCategories());
		
		long sTime = System.nanoTime();
		// the first parameter of parseSentence defines the verbosity
		boolean parseres = false;
		try {
			parseres = parser.parseSentence(verbose, tokens);
		} catch (Exception e) {
			// to get the stats anyway (even if there is no parse)
			long estTime = System.nanoTime() - sTime;
			System.err.println("Total parsing time (no parse found) for sentence \"" + sentence + "\": " + (totalTime+estTime)/(Math.pow(10, 9)) + " sec.");
			//----------------------------------------------------
			//kk Interface.error("No parse found.", op);
			return false;
		}
		long estTime = System.nanoTime() - sTime;
		System.err.println("Parsing time: " + (estTime)/(Math.pow(10, 9)) + " sec.");
		if (parseres){
			System.err.println("Sentence \"" + tok.getSentence() + "\" parsed.");
			// for printing the RCG derivation forest (also printed by the parser in verbose mode)
			//System.err.println(parser.printForest());
			
			// 9. Forest extraction
			ForestExtractor extractor = ForestExtractorFactory.getForestExtractor(parser);
			extractor.init(verbose, rcgg, parser.getForestExtractorInitializer());
			long fTime = System.nanoTime();
			extractor.extract();
			long estfTime = System.nanoTime() - fTime;

			if (verbose)
				System.err.println(extractor.printForest());
			
			Document fdoc = ProduceDOM.buildDOMForest(extractor.getForest(), extractor.getStart(), tok.getSentence(), op.getVal("g"), new NameFactory(), null);
			//DerivedTreeViewer.displayTreesfromDOM(sentence, fdoc, grammarDict, true, op.check("w"), op.check("w"), needsAnchoring, slabels, noUtool);
			// 9'. forest XML export
			if (op.check("f")) {
				Document fdoc2 = ProduceDOM.buildDOMForest(extractor.getForest(), extractor.getStart(), tok.getSentence(), op.getVal("g"), new NameFactory(), grammarDict);
				XMLUtilities.writeXML(fdoc2, op.getVal("f"), "tulipa-forest3.dtd,xml", true);
			}
			System.err.println("Forest extraction time: "+(estfTime)/(Math.pow(10, 9))+" sec.");
			// update the time counter
			totalTime += estTime + estfTime;		
 			
			// 10. GUI or XML output of the parses (and dependencies if needed)
  			if (op.check("x")) { // XML output of the derivations!
  				long xmlTime = System.nanoTime();
				Document dparses = DOMderivationBuilder.buildDOMderivation(DerivedTreeViewer.getViewTreesFromDOM(fdoc, grammarDict, false, false, false, needsAnchoring, slabels, noUtool), sentence);
				XMLUtilities.writeXML(dparses, outputfile, "tulipa-parses.dtd,xml", true);
				long estXMLTime = System.nanoTime();
				System.err.println("Parses available (in XML) in " + outputfile + ".");
				estXMLTime = System.nanoTime() - xmlTime;	

				System.err.println("XML production time: "+(estXMLTime)/(Math.pow(10, 9))+" sec.");
				totalTime += estXMLTime;
			} else { // graphical output (default)
				long estDTime = System.nanoTime();
				DerivedTreeViewer.displayTreesfromDOM(sentence, fdoc, grammarDict, true, op.check("w"), op.check("w"), needsAnchoring, slabels, noUtool);
				long dDTime = System.nanoTime() - estDTime;
				System.err.println("Derivation trees extraction time: "+(dDTime)/(Math.pow(10, 9))+" sec.");
				totalTime += dDTime;						
  			}
  			
  			if (op.check("d")) { // dependency output
				// cannot set file and path to dependency output from the jar
				//String deppath = op.getVal("d") + File.separator;
  				// KK
  				//String deppath = System.getProperty("user.dir") + File.separator;
  				// KK
				String depfile = op.getVal("d");//"dependencies.xml";
				// KK
				//String pdffile = "structure-*.pdf";
				// get dependencies
  				DependencyExtractor de = new DependencyExtractor(tokens, fdoc, grammarDict, needsAnchoring);
  				de.processAll();
  				//System.err.println(de.toString());
  				Document ddd = DependencyDOMbuilder.buildAllDep(tokens, sentence, de.getDependences());
  				// KK
  				XMLUtilities.writeXML(ddd, depfile, "http://w3.msi.vxu.se/~nivre/research/MALTXML.dtd,xml", false);
				System.err.println("XML dependency structures available in " + depfile + ".");
				// KK
				//DTool.toPDF(System.getProperty("user.dir") + File.separator + depfile);
				// KK
				//System.err.println("PDF dependency structures available in " + deppath + File.separator + pdffile + ".");
			}
			   			
  			res = true;
  			
		} else {
			long noTime = System.nanoTime() - sTime;
			totalTime += noTime;
			System.err.println("No derivation forest available.");
			res = false;
		}
		
		// total = loading + anchoring + conversion + parsing + forest + XML / derivation trees
		System.err.println("\nTotal parsing time for sentence \""+ sentence + "\": " +(totalTime)/(Math.pow(10, 9))+" sec.");

		return res;
	}

}
