package tag.extract;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;

import parser.Feature;
import parser.SwigraTag;
import parser.SwigraTree;
import trees.SwigraTreeNode;
import trees.TAGTreeNode;
import utility.Counter;
import utility.Pair;

public class TAGExtractor {

	public Set<TAGTreeNode> trees = new HashSet<TAGTreeNode>();
	public Map<TAGTreeNode, Set<Pair<String, String>>> anchors = new HashMap<TAGTreeNode, Set<Pair<String, String>>>();
	public Map<Feature, Set<String>> featureValues = new HashMap<Feature, Set<String>>();
	private int counter;
	private int counter2;
	// private Map<String, Integer> mapping;
	// private Counter<String> featureCounter;
	private Counter<SwigraTag> treeCounter = new Counter<SwigraTag>();
	private static AltTreeBuilder[] builders = { new AgltFromVerb(), new PrzyzloFromVerb(), new VerbFromAglt() };

	public TAGExtractor() {
		for (Feature f : TAGTreeNode.features) {
			this.featureValues.put(f, new TreeSet<String>());
		}
	}

	public void extractTAGTrees(SwigraTree tree) {
		boolean v = /* false && */tree.words().size() == 7
				&& (tree.words().contains("piwoszkę"));
		// && (/*tree.words().contains("ślady") ||
		// tree.words().contains("nabywaniu") ||
		// */tree.words().contains("koszmar"));
		if (v) {
			// System.out.println("\n");
			// System.out.println(tree.sentence() + "\\\\");
			// System.out.println(tree.fullTexTree());
		}
		if (tree.getRoot().children.size() == 4) {
			// System.out.println(tree.sentence());
		}
		this.addTAGTree(this.extract(tree.getRoot(), v), v);
	}

	public void prepareTree(SwigraTree tree) throws Exception {
		if (tree.getRoot().children.size() > 2) {
			List<Integer> sentChildren = new LinkedList<Integer>();
			for (SwigraTreeNode child : tree.getRoot().children) {
				if (child.tag == SwigraTag.ZDANIE) {
					sentChildren.add(child.id);
				}
			}
			if (sentChildren.size() == 1) {
				for (SwigraTreeNode child : tree.getRoot().children) {
					child.isHead = sentChildren.contains(child.id);
				}
				tree.getRoot().headChildren.clear();
				tree.getRoot().headChildren.addAll(sentChildren);
			} else {
				System.out.println(sentChildren);
				System.out.println(tree.sentence());
			}
		}
		this.checkAglt(tree.getRoot());
		if (!this.checkTerminals(tree.getRoot())) {
			throw new Exception("ZNAK");
		}
		this.fixBedzie(tree.getRoot());
		this.addVerbGender(tree.getRoot());
		this.addNounPerson(tree.getRoot());
	}

	private void checkAglt(SwigraTreeNode node) {
		// TODO
		if (node.tag == SwigraTag.FORMA_CZAS && node.children.size() == 2) {
			SwigraTreeNode ch1 = node.children.get(0);
			SwigraTreeNode ch2 = node.children.get(1);
			if (ch1.isTerminal() && ch2.isTerminal()
					&& ch1.fs.get(Feature.POS).contentEquals("praet")
					&& ch2.fs.get(Feature.POS).contentEquals("aglt")) {
				ch1.isHead = true;
				node.children.remove(ch2);
				node.headChildren.clear();
				node.headChildren.add(ch1.id);
				SwigraTreeNode aglt = new SwigraTreeNode();
				aglt.id = 301;
				aglt.tag = SwigraTag.AGLT;
				for (int i = 0; i < TAGTreeNode.grammarFeatures.length; ++i) {
					Feature f = TAGTreeNode.grammarFeatures[i];
					if ((f == Feature.LICZBA || f == Feature.OSOBA) && ch2.fs.containsKey(f)) {
						aglt.fs.put(f, ch2.fs.get(f));
					}
				}
				aglt.children.add(ch2);
				ch2.isHead = true;
				aglt.headChildren.add(ch2.id);
				node.children.add(aglt);
			}
		}
		for (SwigraTreeNode child : node.children) {
			this.checkAglt(child);
		}
	}

	private boolean checkTerminals(SwigraTreeNode node) {
		if (node.isTerminal()
				&& !node.fs.get(Feature.POS).contentEquals("interp")) {
			if (node.base.contains(" ") || node.orth.contains(" ")) {
				return false;
			}
			if (node.base.contains(".") || node.orth.contains(".")) {
				return false;
			}
			if (node.base.contains(",") || node.orth.contains(",")) {
				return false;
			}
			if (node.base.contains("(") || node.orth.contains("(")) {
				return false;
			}
			if (node.base.contains("[") || node.orth.contains("[")) {
				return false;
			}
		}
		for (SwigraTreeNode child : node.children) {
			if (!this.checkTerminals(child)) {
				return false;
			}
		}
		return true;
	}

	// bedzie ma nie być głową + dodajemy pośredni węzeł
	private void fixBedzie(SwigraTreeNode node) {
		if (node.tag == SwigraTag.FORMA_CZAS && node.children.size() == 2) {
			SwigraTreeNode child1 = node.children.get(0);
			SwigraTreeNode child2 = node.children.get(1);
			// będzie + czasownik
			if (child1.isTerminal()
					&& child1.fs.get(Feature.POS).contentEquals("bedzie")
					&& child2.tag == SwigraTag.PRZY_ZLO) {
				System.err.println("bedzie");
				SwigraTreeNode bedzie = new SwigraTreeNode();
				bedzie.id = 201;
				bedzie.tag = SwigraTag.BEDZIE;
				for (int i = 0; i < TAGTreeNode.grammarFeatures.length; ++i) {
					Feature f = TAGTreeNode.grammarFeatures[i];
					if ((f == Feature.LICZBA || f == Feature.OSOBA || f == Feature.ASPEKT)
							&& node.fs.containsKey(f)) {
						bedzie.fs.put(f, node.fs.get(f));
					}
				}
				bedzie.fs.put(Feature.CZAS, "przy");
				bedzie.children.add(child1);
				bedzie.headChildren.add((Integer) child1.id);
				node.children.remove(child1);
				node.headChildren.remove((Integer) child1.id);
				node.children.add(0, bedzie);
				child2.children.get(0).fs.put(Feature.OSOBA, "3");
			}
		}
		for (SwigraTreeNode child : node.children) {
			this.fixBedzie(child);
		}
	}

	private void addVerbGender(SwigraTreeNode node) {
		if (node.tag == SwigraTag.FORMA_CZAS
				&& node.fs.containsKey(Feature.RODZAJ)) {
			for (SwigraTreeNode child : node.children) {
				// TODO hack dla BEDZIE
				if (child.tag != SwigraTag.BEDZIE
						&& !child.fs.containsKey(Feature.RODZAJ)) {
					child.fs.put(Feature.RODZAJ, node.fs.get(Feature.RODZAJ));
				}
			}
		}
		for (SwigraTreeNode child : node.children) {
			this.addVerbGender(child);
		}
	}

	private void addNounPerson(SwigraTreeNode node) {
		if (node.tag == SwigraTag.FNO && node.fs.containsKey(Feature.OSOBA)) {
			String person = node.fs.get(Feature.OSOBA);
			SwigraTreeNode child = node;
			while (!child.isTerminal()) {
				child = child.getFirstHeadChild();
				if (!child.fs.containsKey(Feature.OSOBA)) {
					child.fs.put(Feature.OSOBA, person);
				}
			}
		}
		for (SwigraTreeNode child : node.children) {
			this.addNounPerson(child);
		}
	}

	private TAGTreeNode extract(SwigraTreeNode node, boolean verbose) {
		TAGTreeNode tagNode = new TAGTreeNode(node);
		boolean left = true;
		SwigraTreeNode headChild = null;
		if (verbose) {
			System.out.println("parent: " + node.tag);
		}
		for (SwigraTreeNode child : node.children) {
			if (verbose) {
				System.out.println(child.tag + " " + child.isHead);
			}
			if (child.isHead) {
				headChild = child;
				if (verbose) {
					System.out.println("---> HEAD");
				}
			}
		}
		List<SwigraTreeNode> duals = new LinkedList<SwigraTreeNode>();
		for (SwigraTreeNode child : node.children) {
			if (this.isDual(child)) {
				duals.add(child);
			}
		}
		if (!duals.isEmpty()) {
			if (duals.size() != 2) {
				throw new RuntimeException();
			}
			node.children.removeAll(duals);
			TAGTreeNode auxTree = new TAGTreeNode(headChild);
			auxTree.fs.remove(Feature.TFW);
			TAGTreeNode footNode = new TAGTreeNode(headChild);
			footNode.fs.remove(Feature.TFW);
			footNode.foot = true;
			TAGTreeNode d1 = this.extract(duals.get(0), verbose);
			d1.trunk = true;
			TAGTreeNode d2 = this.extract(duals.get(1), verbose);
			d2.trunk = true;
			d2.getAnchors().get(0).coanchor = true;
			d2.getCoanchors().get(0).name = "coanch";
			auxTree.children.add(d1);
			auxTree.children.add(footNode);
			auxTree.children.add(d2);
			auxTree.headChild = footNode;
			auxTree.footNode = footNode;
			this.addTAGTree(auxTree, verbose);
		}
		for (SwigraTreeNode child : node.children) {
			if (child.isHead) {
				if (verbose) {
					System.out.println(node.tag + ", head child: " + child.tag);
				}
				TAGTreeNode trunk = this.extract(child, verbose);
				trunk.trunk = true;
				tagNode.children.add(trunk);
				tagNode.headChild = trunk;
				left = false;
			} else {
				if (this.isSubst(child, node, headChild)) {
					TAGTreeNode substNode = new TAGTreeNode(child);
					substNode.subst = true;
					if (substNode.isTerminal()) {
						substNode.coanchor = true;
						substNode.name = "coanch";
					}
					tagNode.children.add(substNode);
					this.addTAGTree(this.extract(child, verbose), verbose);
				} else {
					TAGTreeNode auxTree;
					TAGTreeNode footNode;
					if (this.modifiesHead(child, node, headChild)) {
						auxTree = new TAGTreeNode(node);
						footNode = new TAGTreeNode(node);
					} else {
						auxTree = new TAGTreeNode(headChild);
						footNode = new TAGTreeNode(headChild);
					}
					auxTree.fs.remove(Feature.TFW);
					footNode.fs.remove(Feature.TFW);
					footNode.foot = true;
					TAGTreeNode trunk = this.extract(child, verbose);
					trunk.trunk = true;
					if (left) {
						auxTree.children.add(trunk);
						auxTree.children.add(footNode);
					} else {
						auxTree.children.add(footNode);
						auxTree.children.add(trunk);
					}
					auxTree.headChild = trunk;
					auxTree.footNode = footNode;
					this.addTAGTree(auxTree, verbose);
				}
			}
		}
		return tagNode;
	}

	private boolean isDual(SwigraTreeNode child) {
		return (child.tag == SwigraTag.NAWIAS || child.tag == SwigraTag.CUDZ);
	}

	private boolean isSubst(SwigraTreeNode child, SwigraTreeNode parent,
			SwigraTreeNode headSibling) {
		if (child.fs.containsKey(Feature.TFW)) {
			return true;
		}
		if (child.fs.containsKey(Feature.JAKO_FL)) {
			return false;
		}
		if (child.tag == SwigraTag.PAUZA) {
			return false;
		}
		if (child.tag == SwigraTag.SPOJNIK
				&& parent.tag == SwigraTag.WYPOWIEDZENIE) {
			return false;
		}
		if (parent.rule.contentEquals("noap")) {
			// TODO - dla apozycji
			return true;
		}
		if (!parent.tag.equals(headSibling.tag)) {
			// w przeciwnym przypadku "modyfikator"
			return true;
		}
		return false;
	}

	private boolean modifiesHead(SwigraTreeNode child, SwigraTreeNode head,
			SwigraTreeNode headChild) {
		// aglt
		if (child.tag == SwigraTag.AGLT) {
			return true;
		}
		int childIndex = head.children.indexOf(child);
		int headChildIndex = head.children.indexOf(headChild);
		int step = 1;
		if (childIndex > headChildIndex) {
			step = -1;
		}
		int i = childIndex;
		while (i != headChildIndex) {
			if (head.children.get(i).fs.containsKey(Feature.TFW)) {
				// napotkano argument
				return true;
			}
			i += step;
		}
		return false;
	}

	private List<TAGTreeNode> altTAGTrees(TAGTreeNode tree) {
		List<TAGTreeNode> ret = new LinkedList<TAGTreeNode>();
		TAGTreeNode formaczas = tree.findFirst(SwigraTag.FORMA_CZAS);
		if (formaczas != null) {
			//System.out.println(tree.toTex());
			TAGTreeNode t1 = null, t2 = null;
			TAGTreeNode child1 = formaczas.children.get(0);
			if (formaczas.children.size() == 1 && child1.isTerminal() && child1.fs.get(Feature.POS).contentEquals("verb")) {
				t1 = (new AgltFromVerb().build(tree));
				if (child1.fs.get(Feature.ASPEKT).contentEquals("imperf")) {
					t2 = (new PrzyzloFromVerb().build(tree));
				}
			} else if (tree.children.size() == 2 && child1.tag == SwigraTag.BEDZIE) {
				t1 = (new VerbFromPrzyzlo().build(tree));
				t2 = (new AgltFromVerb().build(t1));
			} else if (tree.children.size() == 2 && tree.children.get(1).tag == SwigraTag.AGLT){
				t1 = (new VerbFromAglt().build(tree));
				t2 = (new PrzyzloFromVerb().build(t1));
			}
			if (t1 != null) {
				this.fixVerbFeatures(t1);
				ret.add(t1);
				//System.out.println(t1.toTex());
			}
			if (t2 != null) {
				this.fixVerbFeatures(t2);
				ret.add(t2);
				//System.out.println(t2.toTex());
			}
		}
		return ret;
	}

	// jeśli drzewo zostało stworzone jako alternatywne, trzeba poprawić
	// uzgodnienia
	private void fixVerbFeatures(TAGTreeNode tree) {
		TAGTreeNode bedzie = tree.findFirst(SwigraTag.BEDZIE);
		if (bedzie != null) {
			// forma złożona - uzgodnienie z bedzie
			// węzeł, z którego bierzemy wartości do zastąpienia;
			// jeśli jest to WYPOWIEDZENIE (bez atrybutów), to bierzemy ZDANIE
			// System.out.println(tree.toTex());
			TAGTreeNode refNode = tree.findFirst(SwigraTag.FF);
			for (Entry<Feature, String> e : bedzie.fs.entrySet()) {
				Feature f = e.getKey();
				this.replaceFeatureValue(tree, f, refNode.fs.get(f),
						e.getValue());
			}
			// System.out.println(tree.toTex());
			// System.exit(1);
		} else {
			// TODO - formaczas uzgodniona z reszta
		}
	}

	private void addTAGTree(TAGTreeNode tree, boolean verbose) {
		// System.out.println("TAGExtractor.addTAGTree");
		// System.out.println(tree.toTex());
		List<TAGTreeNode> alt = this.altTAGTrees(tree);
		for (TAGTreeNode t : alt) {
			// System.err.println("TAGExtractor.addTAGTree");
			//this.newTAGTree(t, verbose);
		}
		this.newTAGTree(tree, verbose);
	}

	private void newTAGTree(TAGTreeNode tree, boolean verbose) {
		/*
		 * System.err.println("TAGExtractor.newTAGTree");
		 * System.out.println(tree.tag); System.out.println(tree.children); for
		 * (TAGTreeNode ch : tree.children) { System.out.println(" " + ch.tag);
		 * for (TAGTreeNode chch : ch.children) { System.out.println("  " +
		 * chch.tag); for (TAGTreeNode chchch : chch.children) {
		 * System.out.println("   " + chchch.tag); for (TAGTreeNode chchchch :
		 * chchch.children) { System.out.println("    " + chchchch.tag); for
		 * (TAGTreeNode chchchchch : chchchch.children) {
		 * System.out.println("     " + chchchchch.tag); } } } } }
		 * System.out.println(tree.shape());
		 */
		tree.trunk = true;
		this.collectFeatureValues(tree);
		tree.reduce();
		this.giveIds(tree);
		this.unify(tree);
		tree.clean();
		List<String> eqs = tree.getEquations();
		tree.treeId = this.treeCounter.increaseAndGet(tree.tag);
		Set<Pair<String, String>> anchorsLex = new HashSet<Pair<String, String>>();
		for (TAGTreeNode a : tree.getAnchors()) {
			try {
				String s = "*EQUATIONS:\n";
				for (String eq : eqs) {
					s += "    " + eq + "\n";
				}
				s += "*COANCHORS:\n";
				for (TAGTreeNode coanch : tree.getCoanchors()) {
					s += "    " + coanch.name + " -> " + coanch.makeBase()
							+ "/" + coanch.fs.get(Feature.POS) + "\n";
				}
				anchorsLex.add(new Pair<String, String>(a.lexEntry(), s));
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		/*
		 * if (tree.treeId().contentEquals("FNOAUX_10a")) {
		 * System.out.println(tree.toMG()); }
		 */
		if (!this.anchors.containsKey(tree)) {
			this.anchors.put(tree, new HashSet<Pair<String, String>>());
		}
		this.anchors.get(tree).addAll(anchorsLex);
		if (verbose) {
			// System.out.println(tree.toTex());
		}
		this.trees.add(tree);
	}

	private void giveIds(TAGTreeNode tree) {
		this.counter = 0;
		this.counter2 = 0;
		this.ids(tree);
		tree.maxId = this.counter;
	}

	private void ids(TAGTreeNode node) {
		node.id = (++this.counter);
		if (node.subst) {
			node.name = "substNode_" + (++this.counter2);
		}
		for (TAGTreeNode child : node.children) {
			this.ids(child);
		}
	}

	private void unify(TAGTreeNode tree) {
		this.counter = 0;
		// this.featureCounter = new Counter<String>();
		this.unifyHelper(tree, true);
		tree.maxVar = this.counter;
	}

	private void unifyHelper(TAGTreeNode node, boolean isRoot) {

		if (node.children.isEmpty()) {
			return;
		}
		for (Feature f : TAGTreeNode.unifyFeatures) {
			String value = node.headChild.fs.get(f);
			if (value != null && !value.startsWith("?V_")) {
				boolean unify = false;
				if (node.fs.containsKey(f)
						&& node.fs.get(f).contentEquals(value)) {
					unify = true;
				}
				if (value.contentEquals("dop") && !isRoot) {
					// TODO - dopelniacz - nie można stracic info o przypadku
					unify = false;
				}
				if (unify) {
					String var = "?V_" + (++this.counter);
					for (TAGTreeNode child : node.children) {
						if (this.unifyChild(child, node, node.headChild, f)) {
							this.replaceFeatureValue(child, f, value, var);
						}
					}
					node.fs.put(f, var);
				}
			}
		}
		for (TAGTreeNode child : node.children) {
			if (child != node.headChild) {
				this.varChildFeatures(child, node, node.headChild);
			}
		}
		if (node.isAux()) {
			for (Feature f : node.fs.keySet()) {
				if (!node.fs.get(f).startsWith("?V_")) {
					node.fs.put(f, "?V_" + (++this.counter));
				}
			}
			node.footNode.fs.putAll(node.fs);
		}
		this.unifyHelper(node.headChild, false);
	}

	private boolean unifyChild(TAGTreeNode child, TAGTreeNode parent,
			TAGTreeNode headSibling, Feature feature) {
		if (child == parent.headChild) {
			return true;
		}
		if (!child.fs.containsKey(feature)) {
			return false;
		}
		if (child.tag == parent.tag) {
			// TODO - dodane dla apozycji
			// w innym przypadku nie powinno byc takiej sytuacji
			return true;
		}
		if (child.fs.containsKey(Feature.TFW)) {
			String tfw = child.fs.get(Feature.TFW);
			if (tfw.contentEquals("subj")) {
				return (feature == Feature.LICZBA || feature == Feature.RODZAJ || feature == Feature.OSOBA);
			}
			int begin = tfw.indexOf("(");
			String[] TFWfs = tfw.substring((begin > 0 ? begin + 1 : 0),
					tfw.length() - 1).split(", ");
			for (String s : TFWfs) {
				if (child.fs.get(feature).contentEquals(s)) {
					return false;
				}
			}
			return (child.tag == SwigraTag.FPT);
		}
		return (child.tag == SwigraTag.BEDZIE || child.tag == SwigraTag.FPT || parent.tag == SwigraTag.FPM);
	}

	private void varChildFeatures(TAGTreeNode child, TAGTreeNode parent,
			TAGTreeNode headSibling) {
		if (child.fs.containsKey(Feature.TFW)) {
			String tfw = child.fs.get(Feature.TFW);
			if (tfw.contentEquals("subj")) {
				tfw += "(mian)";
			}
			int begin = tfw.indexOf("(");
			String[] TFWfs = tfw.substring((begin > 0 ? begin + 1 : 0),
					tfw.length() - 1).split(", ");
			for (Feature f : child.fs.keySet()) {
				if (!child.fs.get(f).startsWith("?V_")) {
					boolean required = false;
					for (String s : TFWfs) {
						if (child.fs.get(f).contentEquals(s)) {
							required = true;
						}
						// TODO - wyjatkowo szpetna lata na np(bier) w
						// zaprzeczeniu
						if (s.contentEquals("bier")
								&& child.fs.get(f).contentEquals("dop")) {
							required = true;
						}
					}
					// TODO - zmieniona obsluga rekcji (przez equations)
					if (required) {
						child.equations.add(child.name + " -> " + f.getFname()
								+ " = " + child.fs.get(f));
					}
					if (f != Feature.POS) {
						child.fs.put(f, "?V_" + (++this.counter));
					}
					/*
					 * if (!required) { child.fs.put(f, "?V_" +
					 * (++this.counter)); }
					 */
				}
			}
		} else {
			for (Feature f : child.fs.keySet()) {
				if (f != Feature.PRZYPADEK && f != Feature.POS
						&& !child.fs.get(f).startsWith("?V_")) {
					child.fs.put(f, "?V_" + (++this.counter));
				}
			}
		}
	}

	private void old_unifyHelper(TAGTreeNode node) {
		/*
		 * for (Feature f : TAGTreeNode.features) { if (node.fs.containsKey(f))
		 * { String value = node.fs.get(f); this.featureCounter.increase(value);
		 * if (this.featureCounter.get(value) > 1 &&
		 * !this.mapping.containsKey(value)) { this.mapping.put(value,
		 * ++this.counter); } } } for (TAGTreeNode child : node.children) {
		 * this.unifyHelper(child); } for (Feature f : TAGTreeNode.features) {
		 * if (node.fs.containsKey(f) && this.featureCounter.get(node.fs.get(f))
		 * > 1) { node.fs.put(f, "?V_" + this.mapping.get(node.fs.get(f))); } }
		 * node.maxVar = this.counter;
		 */

		/*
		 * for (Feature f : TAGTreeNode.features) { Map<String, Integer> mapping
		 * = new HashMap<String, Integer>(); Counter<String> counter = new
		 * Counter<String>(); for (TAGTreeNode child : node.children) { if
		 * (child.fs.containsKey(f)) { counter.increase(child.fs.get(f)); } }
		 * for (TAGTreeNode child : node.children) { String value =
		 * child.fs.get(f); if (value != null && counter.get(value) > 1) { if
		 * (!mapping.containsKey(value)) { mapping.put(value, ++this.counter); }
		 * child.replaceFeatureValue(f, value, "?V_" + mapping.get(value)); } }
		 * if (node.fs.containsKey(f) && mapping.containsKey(node.fs.get(f))) {
		 * node.fs.put(f, "?V_" + mapping.get(node.fs.get(f))); } }
		 */

		/*
		 * for (TAGTreeNode child : node.children) { this.unifyHelper(child); }
		 */

		if (node.children.isEmpty()) {
			return;
		}
		for (Feature f : TAGTreeNode.unifyFeatures) {
			String value = node.headChild.fs.get(f);
			if (value != null && !value.startsWith("?V_")) {
				boolean unify = false;
				if (node.fs.containsKey(f)
						&& node.fs.get(f).contentEquals(value)) {
					unify = true;
				}
				for (TAGTreeNode child : node.children) {
					if (!child.trunk && this.unifyChild(child, node.headChild)
							&& child.fs.containsKey(f)
							&& child.fs.get(f).contentEquals(value)) {
						unify = true;
					}
				}
				if (unify) {
					this.replaceFeatureValue(node, f, value, "?V_"
							+ (++this.counter));
				}
			}
		}
		// if (node.headChild.fs.containsKey(Feature.REKCJA)) {
		// System.out.println(node.headChild.fs.get(Feature.REKCJA));
		for (TAGTreeNode child : node.children) {
			if (child.fs.containsKey(Feature.TFW)) {
				String tfw = child.fs.get(Feature.TFW);
				if (tfw.indexOf("(") > 0) {
					String[] TFWfs = tfw.substring(tfw.indexOf("(") + 1,
							tfw.length() - 1).split(", ");
					for (Feature f : child.fs.keySet()) {
						boolean required = false;
						for (String val : TFWfs) {
							if (child.fs.get(f).contentEquals(val)) {
								required = true;
							}
						}
						if (!required && child.tag != SwigraTag.FPT) {
							child.fs.put(f, "?V_" + (++this.counter));
						}
					}
				}
				if (tfw.contentEquals("subj")) {
					Feature[] flist = { Feature.LICZBA, Feature.RODZAJ,
							Feature.OSOBA };
					for (Feature f : flist) {
						child.fs.put(f, node.headChild.fs.get(f));
					}
				}
			}
		}
		// }
		if (node.isAux()) {
			node.footNode.fs.putAll(node.fs);
		}
		this.old_unifyHelper(node.headChild);
	}

	private void replaceFeatureValue(TAGTreeNode node, Feature f,
			String oldVal, String newVal) {
		boolean bedzie = false && (node.tag == SwigraTag.BEDZIE && f == Feature.CZAS);
		if (bedzie) {
			System.err.println("TAGExtractor.replaceFeatureValue --- 1");
			System.err.println(f + " " + oldVal + " " + newVal);
		}
		if (node.fs.containsKey(f) && node.fs.get(f).contentEquals(oldVal)) {
			if (bedzie) {
				System.err.println("REPLACING: " + f + " " + oldVal + " "
						+ newVal);
			}
			node.fs.put(f, newVal);
		}
		if (bedzie) {
			System.err.println("TAGExtractor.replaceFeatureValue --- 2");
		}
		for (TAGTreeNode child : node.children) {
			// if (this.unifyChild(child, node.headChild)) {
			// TODO hack dla PRZY_ZLO
			if (node.tag != SwigraTag.PRZY_ZLO
					|| (f != Feature.OSOBA && f != Feature.CZAS)) {
				this.replaceFeatureValue(child, f, oldVal, newVal);
			}
			// }
		}
	}

	private boolean unifyChild(TAGTreeNode node, TAGTreeNode headSibling) {
		if (node.fs.containsKey(Feature.TFW) && node.tag != SwigraTag.FPT) {
			return false;
		}
		if (!node.trunk && node.tag == SwigraTag.FNO) {
			return false;
		}
		return true;
	}

	private void collectFeatureValues(TAGTreeNode node) {
		for (Feature f : TAGTreeNode.features) {
			if (node.fs.containsKey(f)) {
				try {
					this.featureValues.get(f).add(node.fs.get(f));
				} catch (NullPointerException e) {
					System.err.println("TAGExtractor.collectFeatureValues "
							+ node.tag);
					System.err
							.println("TAGExtractor.collectFeatureValues " + f);
					System.err.println("TAGExtractor.collectFeatureValues "
							+ node.fs.get(f));
					throw e;
				}
			}
		}
		for (TAGTreeNode child : node.children) {
			this.collectFeatureValues(child);
		}
	}

	private boolean hasRecursion(TAGTreeNode node) {
		if (node.isAux()) {
			return false;
		}
		return this.hasRecursion(node, new HashSet<SwigraTag>());
	}

	private boolean hasRecursion(TAGTreeNode node,
			Collection<SwigraTag> trunkTags) {
		if (node.children.isEmpty()) {
			return false;
		}
		for (TAGTreeNode child : node.children) {
			if (!child.trunk && trunkTags.contains(child.tag)) {
				return true;
			}
		}
		trunkTags.add(node.tag);
		return this.hasRecursion(node.headChild, trunkTags);
	}

}
