package trees;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import parser.Feature;
import parser.SwigraTag;
import utility.Utility;

public class TAGTreeNode extends TreeNode<TAGTreeNode> {

	private static final long serialVersionUID = -4427642565196296419L;
	public boolean subst = false;
	public boolean foot = false;
	public boolean trunk = false;
	// dla nawiasow i cudzyslowow
	public boolean coanchor = false;
	public TAGTreeNode headChild;
	public TAGTreeNode footNode;
	public int maxId = 0;
	public int maxVar = 0;
	public int treeId = 0;
	public String name = "";
	public List<String> equations = new LinkedList<String>();
	public static Feature[] features = { Feature.POS, Feature.LICZBA,
			Feature.PRZYPADEK, Feature.RODZAJ, Feature.OSOBA, Feature.CZAS,
			Feature.REKCJA, Feature.TFW, Feature.ASPEKT };
	public static Feature[] grammarFeatures = { Feature.POS, Feature.LICZBA,
			Feature.PRZYPADEK, Feature.RODZAJ, Feature.OSOBA, Feature.CZAS };
	public static Feature[] unifyFeatures = { Feature.LICZBA,
			Feature.PRZYPADEK, Feature.RODZAJ, Feature.OSOBA, Feature.CZAS };
	private static Feature[] tempFeatures = { Feature.REKCJA, Feature.TFW,
			Feature.ASPEKT };

	public TAGTreeNode() {
		this.children = new ArrayList<TAGTreeNode>();
	}

	public TAGTreeNode(SwigraTreeNode node) {
		this();
		this.tag = node.tag;
		this.base = node.base;
		this.orth = node.orth;
		for (Feature f : TAGTreeNode.features) {
			if (node.fs.containsKey(f) && !node.fs.get(f).contentEquals("_")) {
				String val = Utility.toAscii(node.fs.get(f));
				if (f == Feature.POS) {
					if (val.contentEquals("praet") || val.contentEquals("fin")) {
						val = "verb";
					}
					if (val.contentEquals("psubst")) {
						val = "subst";
					}
				}
				this.fs.put(f, val);
			}
		}
		// dodajemy czas i osobę!!!
		if (node.fs.containsKey(Feature.POS)) {
			String pos = node.fs.get(Feature.POS);
			if (pos.contentEquals("fin")) {
				if (node.fs.get(Feature.ASPEKT).contentEquals("perf")) {
					this.fs.put(Feature.CZAS, "przy");
				} else {
					this.fs.put(Feature.CZAS, "ter");
				}
			}
			if (pos.contentEquals("bedzie")) {
				this.fs.put(Feature.CZAS, "przy");
			}
			if (pos.contentEquals("praet")) {
				this.fs.put(Feature.CZAS, "prze");
				this.fs.put(Feature.OSOBA, "3");
			}
		}
	}

	@Override
	public boolean equals(Object obj) {
		if (obj.getClass() != this.getClass()) {
			return false;
		}
		TAGTreeNode other = (TAGTreeNode) obj;
		if (this.tag != other.tag || this.foot != other.foot
				|| this.subst != other.subst) {
			return false;
		}
		/*
		 * for (Feature f : TAGTreeNode.grammarFeatures) { if
		 * (this.fs.containsKey(f) != other.fs.containsKey(f)) { return false; }
		 * if (this.fs.containsKey(f) &&
		 * !this.fs.get(f).contentEquals(other.fs.get(f))) { return false; } }
		 */
		return this.fs.equals(other.fs) && this.children.equals(other.children);
	}

	@Override
	public int hashCode() {
		int hash = (this.tag.hashCode() + (this.foot ? 1 : 0) + (this.subst ? 1
				: 0));
		return (hash + this.fs.hashCode() + this.children.hashCode());
	}

	public TAGTreeNode nodeCopy() {
		TAGTreeNode copy = new TAGTreeNode();
		copy.tag = this.tag;
		for (Feature f : TAGTreeNode.features) {
			if (this.fs.containsKey(f)) {
				copy.fs.put(f, this.fs.get(f));
			}
		}
		copy.base = this.base;
		copy.orth = this.orth;
		copy.subst = this.subst;
		copy.foot = this.foot;
		copy.coanchor = this.coanchor;
		copy.trunk = this.trunk;
		copy.id = this.id;
		return copy;
	}

	public void clean() {
		for (Feature f : TAGTreeNode.tempFeatures) {
			this.fs.remove(f);
		}
		for (TAGTreeNode child : this.children) {
			child.clean();
		}
	}

	public void reduce() {
		if (this.children.size() == 1 && this.tag == this.headChild.tag) {
			if (this.foot == this.headChild.foot
					&& this.subst == this.headChild.subst
			/* && this.fs.equals(this.headChild.fs) */) {
				this.children = this.headChild.children;
				this.headChild = this.headChild.headChild;
			} else {
				System.out.println("TAGTreeNode.reduce problem "
						+ (this.foot == this.headChild.foot) + " "
						+ (this.subst == this.headChild.subst) + " "
						+ (this.fs.equals(this.headChild.fs)));
				System.out.println("    " + this.fs);
				System.out.println("    " + this.headChild.fs);
			}
		}
		for (TAGTreeNode child : this.children) {
			child.reduce();
		}
	}

	public String toMG() {
		StringBuilder mg = new StringBuilder("class ");
		mg.append(this.treeId());
		mg.append("\n");
		mg.append("declare");
		for (int i = 1; i <= this.maxId; ++i) {
			mg.append(" ?N_" + i);
		}
		for (int i = 1; i <= this.maxVar; ++i) {
			mg.append(" ?V_" + i);
		}
		mg.append("\n");
		mg.append("{ <syn> {\n");
		mg.append(this.toMGHelper("    "));
		return mg + "}\n}\n\n";
	}

	private String toMGHelper(String tab) {
		// System.err.println("TAGTreeNode.toMGHelper ---");
		StringBuilder mg = new StringBuilder(tab + "node ?N_" + this.id);
		int marks = 0;
		if (this.coanchor) {
			mg.append(" (mark = coanchor, name = " + this.name + ")");
			// System.err.println("TAGTreeNode.toMGHelper coanchor");
			++marks;
		} else {
			if (this.subst) {
				mg.append(" (mark = subst, name = " + this.name + ")");
				// System.err.println("TAGTreeNode.toMGHelper subst");
				++marks;
			}
			if (this.foot) {
				mg.append(" (mark = foot)");
				// System.err.println("TAGTreeNode.toMGHelper foot");
				++marks;
			}
			if (this.tag == SwigraTag.LEAF) {
				mg.append(" (mark = anchor)");
				// System.err.println("TAGTreeNode.toMGHelper anchor");
				++marks;
			}
		}
		if (marks > 1) {
			System.err.println("TAGTreeNode.toMGHelper problem: " + marks
					+ " marks in " + this.tag + " " + this.fs.get(Feature.POS));
			System.exit(1);
		}
		mg.append(" [");
		if (this.tag != SwigraTag.LEAF) {
			mg.append("cat = " + this.tag);
		}
		for (Feature f : TAGTreeNode.grammarFeatures) {
			if (this.fs.containsKey(f)) {
				mg.append((f == Feature.POS ? "cat" : ", " + f.getFname())
						+ " = " + Utility.toAscii(this.fs.get(f)));
			}
		}
		mg.append("]");
		if (!this.children.isEmpty()) {
			mg.append(" {\n");
		}
		for (TAGTreeNode child : this.children) {
			mg.append(child.toMGHelper(tab + "    "));
		}
		if (!this.children.isEmpty()) {
			mg.append(tab + "}");
		}
		mg.append("\n");
		return mg.toString();
	}

	public String toTex() {
		StringBuilder tex = new StringBuilder("\\branchheight{1.2in}\\synttree");
		tex.append(this.toTexHelper("    "));
		return tex.toString().replace("_", "\\_") + "\n";
	}

	private String toTexHelper(String tab) {
		StringBuilder tex = new StringBuilder("\n" + tab);
		tex.append("[ \\fbox{\\begin{tabular}{ l l }");
		if (this.subst && this.tag == SwigraTag.LEAF) {
			tex.append("\\multicolumn{2}{c}{\\color{red}$\\downarrow_2$} \\\\");
		} else {
			if (this.subst) {
				tex.append("\\multicolumn{2}{c}{\\color{red}$\\downarrow$} \\\\");
			}
			if (this.foot) {
				tex.append("\\multicolumn{2}{c}{\\color{red}$\\ast$} \\\\");
			}
			if (this.tag == SwigraTag.LEAF) {
				tex.append("\\multicolumn{2}{c}{\\color{red}$\\diamond"
						+ (this.coanchor ? "\\diamond" : "") + "$} \\\\");
			}
		}
		if (this.tag != SwigraTag.LEAF) {
			tex.append("cat & " + this.tag + "\\\\");
		}
		for (Feature f : TAGTreeNode.grammarFeatures) {
			if (this.fs.containsKey(f)) {
				tex.append((f == Feature.POS ? "cat" : f.getFname()) + " & "
						+ Utility.toAscii(this.fs.get(f)) + "\\\\");
			}
		}
		tex.append("\\end{tabular}}");
		for (TAGTreeNode child : this.children) {
			tex.append(child.toTexHelper(tab + "    "));
		}
		tex.append("\n" + tab + "]");
		return tex.toString();
	}

	public boolean isAux() {
		if (this.foot) {
			return true;
		}
		for (TAGTreeNode child : this.children) {
			if (child.isAux()) {
				return true;
			}
		}
		return false;
	}

	public String treeId() {
		return this.tag + (this.isAux() ? "AUX" : "") + "_"
				+ Integer.toHexString(this.treeId);
	}

	public String lexEntry() throws Exception {
		if (this.tag == SwigraTag.LEAF) {
			String b = this.makeBase();
			return "*ENTRY: " + b + "\n*CAT: " + this.fs.get(Feature.POS)
					+ "\n";
		} else {
			return this.headChild.lexEntry();
		}
	}

	public String morphEntry() throws Exception {
		if (this.tag == SwigraTag.LEAF) {
			String morph = this.makeOrth() + "    " + this.makeBase() + "    [";
			for (Feature f : TAGTreeNode.features) {
				if (this.fs.containsKey(f)) {
					morph += f.getFname() + " = " + this.fs.get(f) + "; ";
				}
			}
			return morph + "]\n";
		} else {
			return this.headChild.morphEntry();
		}
	}

	public List<TAGTreeNode> getAnchors() {
		List<TAGTreeNode> ret = new LinkedList<TAGTreeNode>();
		if (this.tag == SwigraTag.LEAF && !this.coanchor) {
			ret.add(this);
		} else {
			for (TAGTreeNode child : this.children) {
				ret.addAll(child.getAnchors());
			}
		}
		return ret;
	}

	public List<TAGTreeNode> getCoanchors() {
		List<TAGTreeNode> ret = new LinkedList<TAGTreeNode>();
		if (this.coanchor) {
			ret.add(this);
		} else {
			for (TAGTreeNode child : this.children) {
				ret.addAll(child.getCoanchors());
			}
		}
		return ret;
	}

	public List<TAGTreeNode> getAnchorCopies() {
		List<TAGTreeNode> ret = new LinkedList<TAGTreeNode>();
		if (this.tag == SwigraTag.LEAF) {
			// System.out.println(this.orth);
			ret.add(this.nodeCopy());
		} else {
			for (TAGTreeNode child : this.children) {
				ret.addAll(child.getAnchorCopies());
			}
		}
		return ret;
	}

	public List<String> getEquations() {
		List<String> ret = new LinkedList<String>();
		ret.addAll(this.equations);
		for (TAGTreeNode child : this.children) {
			ret.addAll(child.getEquations());
		}
		return ret;
	}

	private boolean validEntry() {
		if (this.fs.get(Feature.POS).contentEquals("interp")) {
			return true;
		}
		if (this.base.contains(" ") || this.orth.contains(" ")) {
			return false;
		}
		if (this.base.contains(".") || this.orth.contains(".")) {
			return false;
		}
		if (this.base.contains(",") || this.orth.contains(",")) {
			return false;
		}
		if (this.base.contains("(") || this.orth.contains("(")) {
			return false;
		}
		return true;
	}

	public String makeBase() {
		for (int i = 0; i < Utility.punctuation.length; ++i) {
			if (this.base.equals((String) Utility.punctuation[i].first())) {
				return (String) Utility.punctuation[i].second();
			}
		}
		return this.base;
	}

	private String makeOrth() {
		for (int i = 0; i < Utility.punctuation.length; ++i) {
			if (this.base.equals((String) Utility.punctuation[i].first())) {
				return (String) Utility.punctuation[i].second();
			}
		}
		return this.orth.toLowerCase();
	}

	@Override
	public int compareTo(TAGTreeNode o) {
		throw new UnsupportedOperationException();
	}

}
